import asyncio
import importlib
import os
import fnmatch
import time as t
from datetime import datetime as dt
from py_pli.pylib import EndPointType
from py_pli.pylib import URPCFunctions
from py_pli.pylib import VUnits
from py_pli.pylib import send_msg
from urpc_enum.corexymoverparameter import CoreXYMoverParameter
from urpc_enum.measurementparameter import MeasurementParameter
from urpc_enum.moverparameter import MoverParameter
from urpc_enum.nodeparameter import NodeParameter
from virtualunits.vu_measurement_unit import VUMeasurementUnit
from virtualunits.meas_seq_generator import meas_seq_generator
from virtualunits.meas_seq_generator import TriggerSignal
from virtualunits.meas_seq_generator import OutputSignal
import config_enum.scan_table_enum as scan_table_enum
import fleming.mleuschner.hw_test1 as hw

# 2021-01-11/Mirko Leuschner
# Alpha Test at Kiel, First Light
# rework1 210503
# rework2 210805

instrument = "Swansea"

# two channel vs one channel
dual = True
#dual = False

#maximum high voltage for PMT
hvmax=188

as1pos=-1
as2pos=-1
bldpos=-1
elspos=-1
fmpos=-1
fmspos=-1

bldpd = 355.0
bldtop = 90
bldbottom = 35.25

elsp1 = 24 #330.3 #23.8 #338.8
elsp2 = 204 #152.5 #203.8 #162.5 #153.0
els_al = 249 #111.375 #245.85

fmsdiag = 37


# Report Functions

rfp = __file__

def reportpath():
    path, filename = os.path.split(__file__)
    md1=dt.now().strftime('%y%m%d')
    fpath = os.path.join(path, instrument+md1)   #Report File Path
    if not os.path.isdir(fpath):
        os.mkdir(fpath)
    return(os.path.join(fpath, 'Report'+md1+'.txt'))

def datapath(fl='Test'):
    global rfp
    path, filename = os.path.split(rfp)
    md1=dt.now().strftime('%y%m%d')
    fpath = path+'/'   #Output File Path
    fl =fl+md1+'_Run'
    no = len(fnmatch.filter(os.listdir(fpath), fl+'*.txt'))+1
    fname1 = str.format(fl+'{0:03d}.txt', no)
    return(fpath+fname1)

def report(text='', source='Note'):
    global rfp
    mt1=dt.now().strftime('%H:%M:%S')
    md1=dt.now().strftime('%y%m%d')
    with open(rfp, 'a') as report:
        report.write('\n'+mt1+'; '+md1+'; '+source+'; '+text+'\n')

# Init Functions

async def init():
    print("Startup Hardware \n")
    await VUnits.instance.hal.StartupHardware()
    print("Hardware started \n")
    await VUnits.instance.hal.InitializeDevice()
    print("Device initialized.\n")
    await VUnits.instance.hal.HomeMovers()
    print("Movers homed.\n")
    await VUnits.instance.hal.TurnLedsOff()
    print("LEDs off.\n")

# Mover Functions

async def as1_home():
    global as1pos
    as1 = VUnits.instance.hal.detectorApertureSlider1
    await as1.Home()
    as_err = as1.Mover.LastHomeStepErrors
    print(f"ApertureSlider 1 Step Errors: {as_err}")
    as1pos=0

async def as1(pos):
    # Move ApertureSlider to absolute position
    global as1pos
    as1 = VUnits.instance.hal.detectorApertureSlider1
    await as1.UseProfile(1)
    await as1.Move(pos)
    print(f"Aperture Position: {pos}")
    as1pos=pos

async def as1_1():
    # Aperture 1,6mm
    await as1(4.4)

async def as1_close():
    # close position
    await as1(0.5)

async def as1_3():
    # Aperture 3mm
    await as1(-4.2)

async def as2_home():
    global as2pos
    as2 = VUnits.instance.hal.detectorApertureSlider2
    await as2.Home()
    as_err = as2.Mover.LastHomeStepErrors
    print(f"ApertureSlider 2 Step Errors: {as_err}")
    as2pos=0

async def as2(pos):
    # Move ApertureSlider to absolute position
    global as2pos
    as2 = VUnits.instance.hal.detectorApertureSlider2
    await as2.UseProfile(1)
    await as2.Move(pos)
    print(f"Aperture Position: {pos}")
    as2pos=pos

async def as2_1():
    # Aperture 1,6mm
    await as2(4.4)

async def as2_close():
    # the close position
    await as2(0.5)

async def as2_3():
    # Aperture 3mm
    await as2(-4.2)

async def asx_home():
    await as1_home()
    await as2_home()

async def asx(pos):
    await as1(pos)
    await as2(pos)

async def asx_1():
    await as1_1()
    await as2_1()

async def asx_close():
    await as1_close()
    await as2_close()

async def asx_3():
    await as1_3()
    await as2_3()

async def bld_home():
    global bldpos
    bld = VUnits.instance.hal.bottomLightDirector
    stepErrors = await bld.Home()    
    print(f"BottomLightDirector homed, Step-Error: {stepErrors}")
    bldpos=0

async def bld(pos):
    # move BLD to position
    global bldpos
    bld = VUnits.instance.hal.bottomLightDirector
    await bld.UseProfile(1)
    await bld.Move(pos)
    print(f"Bottom Light Director moved to {pos}")
    bldpos=pos

async def bld_pd():
    await bld(bldpd)
    print("Bottom Light Director moved to ABS PD")

async def bld_bottom():
    await bld(bldbottom)
    print("Bottom Light Director moved to Bottom Reading Position")

async def bld_top():
    await bld(bldtop)
    print("Bottom Light Director moved to Top Reading Position")

async def els_home():
    global elspos
    els = VUnits.instance.hal.excitationLightSelector
    stepErrors = await els.Home()    
    print("ELS homed, Step-Error: " + str(stepErrors))
    elspos=0

async def els(pos):
    # For adjustment
    global elspos
    els = VUnits.instance.hal.excitationLightSelector
    await els.UseProfile(1)
    await els.Move(pos)
    print(f"Exc.LightSel moved to {pos}")
    elspos=pos

async def els_1():
    # Move ELS to increase beam diameter
    await els(elsp1)
    print("Exc.LightSel moved to Flash 1 (increase beam diameter).")

async def els_2():
    # Move ELS to decrease beam diameter
    await els(elsp2)
    print("Exc.LightSel moved to Flash 2 (decrease beam diameter).")

async def els_a():
    # Move ELS to Alpha position
    await els(els_al)
    print("Exc.LightSel moved to Alpha.")

async def fm_home():
    global fmpos
    fm = VUnits.instance.hal.focusMover
    await fm.Home()
    fm_err = fm.Mover.LastHomeStepErrors
    print(f"FocusMover Step Errors: {fm_err}")
    fmpos=0

async def fm(position):
    focus_mover = VUnits.instance.hal.focusMover
    await focus_mover.UseProfile(1)
    await focus_mover.Move(position)
    print(f"Focus Mover Position {position}")
    fmpos=position

async def fms_home():
    global fmspos
    fms = VUnits.instance.hal.filterModuleSlider
    await fms.Home()
    fms_err = fms.Mover.LastHomeStepErrors
    print(f"FilterModuleSlider Step Errors: {fms_err}")
    fmspos=0

async def fms(position):
    global fmspos
    filter_slider = VUnits.instance.hal.filterModuleSlider
    await filter_slider.UseProfile(1)
    await filter_slider.Move(position)
    print(f"Filter Module {position}")
    fmspos=position

async def fms_diag():
    await fms(fmsdiag)
    print(f"XeFlash diagnostics")

async def fms_set(filter_nr):
    position = 55.2 + (filter_nr - 1) * 24.0
    await fms(position)
    print(f"Filter Module {filter_nr}")

async def pd_home():
    step_error = await VUnits.instance.hal.plateDoor.Home()
    print(f"PlateDoor homed,Error = {step_error}\n")
    
async def pd_open():
    plate_door = VUnits.instance.hal.plateDoor
    await plate_door.UseProfile(1)    
    await plate_door.Open()    
    print("Plate Door open.")

async def pd_close():
    plate_door = VUnits.instance.hal.plateDoor
    await plate_door.UseProfile(1)    
    await plate_door.Close()    
    print("Plate Door closed.")   

async def st_home():
    st = VUnits.instance.hal.scan_table
    await st.Home()
    print(f"Scan Table X Step Errors: {st.CoreXY.LastHomeStepErrors[0]}")
    print(f"Scan Table Y Step Errors: {st.CoreXY.LastHomeStepErrors[1]}")

async def st_getposition():
    st = VUnits.instance.hal.scan_table
    return (await st.GetPosition())

async def st_move(x=0, y=0):
    st = VUnits.instance.hal.scan_table
    await st.Move(x, y)
    stpos = await st.GetPosition()
    xpos = stpos[0]
    ypos = stpos[1]
    print(f"X/Y Pos: {x}:{y} : {xpos}:{ypos}")
    return stpos
     
async def eject():
    await as1_close()
    await st_move(10, 180) #PreLoadPos
    await pd_open()
    await st_move(10, 300) #RightLoadPos
    report('Plate removed','eject')
    print(f"Remove Plate")
     
async def load():
    await st_move(10, 180) #PreLoadPos
    await pd_close()
    await st_home()
    await as1_3()
    report('Plate loaded','load')
    print(f"Plate loaded")
     
async def st_movetowell(x=1, y=1, xo=0, yo=0):
    st = VUnits.instance.hal.scan_table
    await st.MoveToWell(x, y, xo, yo)
    print(f"Well Pos: {x}:{y}; Offset: {xo}:{yo}")
     
async def st_platetype(pt="1536_EnSpire_Adjusting_Plate"):
    st = VUnits.instance.hal.scan_table
    st.SetPlateType(pt)
    report('PlateType: '+pt,'st_platetype')
    print(f"PlateType: {pt}")
     
async def st_measpos(mp="FBD_Top"):
    st = VUnits.instance.hal.scan_table
    st.SetCurrentMeasPosition(mp)
    report('Meas Pos: '+mp,'st_measpos')
    print(f"Meas Pos: {mp}")
    
# Sequencer Functions

signals = {
    'flash'     : (1 << 23),
    'alpha'     : (1 << 22),
    'ingate2'   : (1 << 17),
    'ingate1'   : (1 << 16),
    'hvgate2'   : (1 << 13),
    'hvgate1'   : (1 << 12),
    'hvon3'     : (1 << 10),
    'hvon2'     : (1 <<  9),
    'hvon1'     : (1 <<  8),
    'rstaux'    : (1 <<  6),
    'rstabs'    : (1 <<  5),
    'rstref'    : (1 <<  4),
    'rstpmt2'   : (1 <<  1),
    'rstpmt1'   : (1 <<  0),
}

triggers = {
    'trf'   : (1 << 8),
    'aux'   : (1 << 6),
    'abs'   : (1 << 5),
    'ref'   : (1 << 4),
    'pmt3'  : (1 << 2),
    'pmt2'  : (1 << 1),
    'pmt1'  : (1 << 0),
}

channels = {
    'pmt1'  : (0 << 24),
    'pmt2'  : (1 << 24),
    'pmt3'  : (2 << 24),
    'pmt4'  : (3 << 24),
    'ref'   : (4 << 24),
    'abs'   : (5 << 24),
    'aux'   : (6 << 24),
}


async def get_results(address, size):
    meas = VUnits.instance.hal.measurementUnit.MeasurementFunctions
    return (await meas.ReadResults(address, size, timeout=5))[:size]

async def get_status():
    meas = VUnits.instance.hal.measurementUnit.MeasurementFunctions
    return (await meas.GetStatus(timeout=1))


async def write_sequence(address, sequence):
    if len(sequence) > 28:
        raise Exception(f"Maximum sequence length of 28 instructions exceeded.")
    meas = VUnits.instance.hal.measurementUnit.MeasurementFunctions
    await meas.WriteSequence(address, len(sequence), sequence, timeout=5)

async def start_sequence(address):
    measurement_unit = VUnits.instance.hal.measurementUnit
    meas = measurement_unit.MeasurementFunctions
    await meas.StartSequence(address, timeout=1)

# PMT Functions

async def aoreset():
    address = 0
    sequence = [
        0x0E000000 | ((4 & 7) << 15) | ((4 & 7) << 12) | ((4 & 7) << 3) | (4 & 7),
        0x00000000,
    ]
    await write_sequence(address, sequence)
    await start_sequence(address)
    await asyncio.sleep(0.1)
    if (await get_status())[0] & 0x01:
        return f"aoreset() passed"
    else:
        return f"aoreset() failed"
    return(0)

async def hvgate1(enable):
    address = 0
    if enable:
        sequence = [
            0x02000000 | signals['hvgate1'],
            0x00000000,
        ]
    else:
        sequence = [
            0x03000000 | signals['hvgate1'],
            0x00000000,
        ]
    await write_sequence(address, sequence)
    await start_sequence(address)
    await asyncio.sleep(0.1)
    if (await get_status())[0] & 0x01:
        return f"seq_trf_loopback() passed"
    else:
        return f"seq_trf_loopback() failed"

async def hvgate2(enable):
    address = 0
    if enable:
        sequence = [
            0x02000000 | signals['hvgate2'],
            0x00000000,
        ]
    else:
        sequence = [
            0x03000000 | signals['hvgate2'],
            0x00000000,
        ]
    await write_sequence(address, sequence)
    await start_sequence(address)
    await asyncio.sleep(0.1)


async def hvpmt1on():
    # Reset Signal to turn on!!!
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    op_id = 'hvpmt1on'
    seq_gen = meas_seq_generator()
    seq_gen.ResetSignals(OutputSignal.HVOnPMT1)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
    await measurement_unit.ExecuteMeasurement(op_id)
    PyLogger.logger.info('=============HV PMT1 On!!!==================')

async def hvpmt2on():
    # Reset Signal to turn on!!!
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    op_id = 'hvpmt2on'
    seq_gen = meas_seq_generator()
    seq_gen.ResetSignals(OutputSignal.HVOnPMT2)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
    await measurement_unit.ExecuteMeasurement(op_id)
    PyLogger.logger.info('=============HV PMT2 On!!!==================')

async def hvpmt1off():
    #Set Signal to turn off!!
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    op_id = 'hvpmt1off'
    seq_gen = meas_seq_generator()
    seq_gen.SetSignals(OutputSignal.HVOnPMT1)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
    await measurement_unit.ExecuteMeasurement(op_id)
    PyLogger.logger.info('=============HV PMT1 Off!!!==================')

async def hvpmt2off():
    #Set Signal to turn off!!
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    op_id = 'hvpmt2off'
    seq_gen = meas_seq_generator()
    seq_gen.SetSignals(OutputSignal.HVOnPMT2)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
    await measurement_unit.ExecuteMeasurement(op_id)
    PyLogger.logger.info('=============HV PMT2 Off!!!==================')

async def hvgate1on():
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    op_id = 'hvgate1on'
    seq_gen = meas_seq_generator()
    seq_gen.SetSignals(OutputSignal.HVGatePMT1)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
    await measurement_unit.ExecuteMeasurement(op_id)

async def hvgate2on():
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    op_id = 'hvgate2on'
    seq_gen = meas_seq_generator()
    seq_gen.SetSignals(OutputSignal.HVGatePMT2)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
    await measurement_unit.ExecuteMeasurement(op_id)

async def hvgate1off():
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    op_id = 'hvgate1off'
    seq_gen = meas_seq_generator()
    seq_gen.ResetSignals(OutputSignal.HVGatePMT1)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
    await measurement_unit.ExecuteMeasurement(op_id)

async def hvgate2off():
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    op_id = 'hvgate2off'
    seq_gen = meas_seq_generator()
    seq_gen.ResetSignals(OutputSignal.HVGatePMT2)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
    await measurement_unit.ExecuteMeasurement(op_id)

async def pmt1setdl(dl):
    # set dl-value
    delay = 0.1
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.PMT1DiscriminatorLevel, dl, timeout=1)
    await asyncio.sleep(delay)
    #PyLogger.logger.info('=============PMT1 DL Set==================')

async def pmt1sethv(hv):
    # set hv-value
    delay = 0.1
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.PMT1HighVoltageSetting, hv, timeout=1)
    await asyncio.sleep(delay)
    #PyLogger.logger.info('=============PMT1 HV Set==================')

async def pmt1hvenable():
    # enable hv
    delay = 0.1
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.PMT1HighVoltageEnable, 1, timeout=1)
    await asyncio.sleep(delay)
    #PyLogger.logger.info('=============PMT1 HV Enabled==================')

async def pmt1hvdisable():
    # enable hv
    delay = 0.1
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.PMT1HighVoltageEnable, 0, timeout=1)
    await asyncio.sleep(delay)
    #PyLogger.logger.info('=============PMT1 HV Disabled==================')

async def pmt2setdl(dl):
    # set dl-value
    delay = 0.1
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.PMT2DiscriminatorLevel, dl, timeout=1)
    await asyncio.sleep(delay)
    #PyLogger.logger.info('=============PMT1 DL Set==================')

async def pmt2sethv(hv):
    # set hv-value
    delay = 0.1
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.PMT2HighVoltageSetting, hv, timeout=1)
    await asyncio.sleep(delay)
    #PyLogger.logger.info('=============PMT1 HV Set==================')

async def pmt2hvenable():
    # enable hv
    delay = 0.1
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.PMT2HighVoltageEnable, 1, timeout=1)
    await asyncio.sleep(delay)
    #PyLogger.logger.info('=============PMT1 HV Enabled==================')

async def pmt2hvdisable():
    # enable hv
    delay = 0.1
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.PMT2HighVoltageEnable, 0, timeout=1)
    await asyncio.sleep(delay)
    #PyLogger.logger.info('=============PMT1 HV Disabled==================')

async def flashsetpower(power, scale=4096):
    # set flash power
    delay = 0.1
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampPower, power/scale, timeout=1)
    #await hw.flash_setpw(power)
    await asyncio.sleep(delay)
    #PyLogger.logger.info('=============Flash Power Set==================')



# PMT Functions

async def dlscan(dlstart=0, dlstop=255, dlstep=1, window=100, pre_cnt_window=100):
    '''
    ml 211025:
    discrscan, dual channel
    '''

    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit

    #pre_cnt_window = 100    # 1µs
    wscale = 100000 // pre_cnt_window
    cnt_window = window * wscale
    window_corse, window_fine = divmod(cnt_window, 65536)

    delay = 0.1
    dacscale1 = 256
    await pmt1sethv(0)
    await pmt2sethv(0)
    await pmt1hvdisable()
    await pmt2hvdisable()
    await hvgate1on()
    await hvgate2on()

    op_id = 'discrscan'
    seq_gen = meas_seq_generator()
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=0)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=1)
    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    seq_gen.PulseCounterControl(channel=0, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    seq_gen.PulseCounterControl(channel=1, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    if (window_corse > 0):
        seq_gen.Loop(window_corse)
        seq_gen.Loop(65536)
        seq_gen.TimerWaitAndRestart(pre_cnt_window)
        seq_gen.PulseCounterControl(channel=0, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
        seq_gen.PulseCounterControl(channel=1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
        seq_gen.LoopEnd()
        seq_gen.LoopEnd()
    if (window_fine > 0):
        seq_gen.Loop(window_fine)
        seq_gen.TimerWaitAndRestart(pre_cnt_window)
        seq_gen.PulseCounterControl(channel=0, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
        seq_gen.PulseCounterControl(channel=1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
        seq_gen.LoopEnd()
    seq_gen.GetPulseCounterResult(channel=0, relative=False, resetCounter=True, cumulative=True, dword=False, addrPos=0, resultPos=0)
    seq_gen.GetPulseCounterResult(channel=1, relative=False, resetCounter=True, cumulative=True, dword=False, addrPos=0, resultPos=1)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    measurement_unit.resultAddresses[op_id] = range(0, 2)
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
    if dlstop>255: 
        dlstop=2555
    if dlstart>(dlstop-dlstep):
        dlstart=dlstop-dlstep
    dfp = datapath('DLScan')
    path, filename = os.path.split(dfp)
    result = str.format("File: {0}; Start: {1}; Stop: {2}; Step: {3}; Window: {4}; PreCount: {5}",filename, dlstart, dlstop, dlstep, window, pre_cnt_window)
    report(result, '.alpha.dlscan')
    result = str.format("File: {0}; wScale: {1}; cntWin: {2}; winCorse: {3}; winFine: {4}",filename, wscale, cnt_window, window_corse, window_fine)
    report(result, '.alpha.dlscan')
    print (filename)
    with open(dfp, 'a') as data:
        time_stamp = dt.now().strftime('%Y%m%d %H:%M:%S')
        data.write(f"dac\tcount1\tcount2\n")
        for i in range(dlstart, dlstop+dlstep, dlstep):
            await pmt1setdl( i/dacscale1)
            await pmt2setdl( i/dacscale1)
            try:
                await measurement_unit.ExecuteMeasurement(op_id)
                results = await measurement_unit.ReadMeasurementValues(op_id)
                #print(results)
                print(f"{i}\t{results[0]}\t{results[1]}")
                data.write(f"{i}\t{results[0]}\t{results[1]}\n")
            except BaseException as ex:
                print(f"darkcount() failed: {ex}")
                data.write(f"darkcount() failed: {ex}\n")
    await hvgate1off()
    await hvgate2off()
    await pmt1hvdisable()
    await pmt2hvdisable()
    report('Done', '.alpha.dlscan')
    print('Done')
    return(0)


async def dlscan2(hv=100, dlstart=0, dlstop=255, dlstep=1, window=100, pre_cnt_window=100):
    '''
    ml 211025:
    discrscan, dual channel
    '''

    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit

    #pre_cnt_window = 100    # 1µs
    wscale = 100000 // pre_cnt_window
    cnt_window = window * wscale
    window_corse, window_fine = divmod(cnt_window, 65536)

    delay = 0.1
    dacscale1 = 256
    await pmt1sethv(hv/256)
    await pmt2sethv(hv/256)
    await pmt1hvenable()
    await pmt2hvenable()
    await hvgate1on()
    await hvgate2on()

    op_id = 'discrscan'
    seq_gen = meas_seq_generator()
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=0)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=1)
    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    seq_gen.PulseCounterControl(channel=0, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    seq_gen.PulseCounterControl(channel=1, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    if (window_corse > 0):
        seq_gen.Loop(window_corse)
        seq_gen.Loop(65536)
        seq_gen.TimerWaitAndRestart(pre_cnt_window)
        seq_gen.PulseCounterControl(channel=0, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
        seq_gen.PulseCounterControl(channel=1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
        seq_gen.LoopEnd()
        seq_gen.LoopEnd()
    if (window_fine > 0):
        seq_gen.Loop(window_fine)
        seq_gen.TimerWaitAndRestart(pre_cnt_window)
        seq_gen.PulseCounterControl(channel=0, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
        seq_gen.PulseCounterControl(channel=1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
        seq_gen.LoopEnd()
    seq_gen.GetPulseCounterResult(channel=0, relative=False, resetCounter=True, cumulative=True, dword=False, addrPos=0, resultPos=0)
    seq_gen.GetPulseCounterResult(channel=1, relative=False, resetCounter=True, cumulative=True, dword=False, addrPos=0, resultPos=1)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    measurement_unit.resultAddresses[op_id] = range(0, 2)
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
    if dlstop>255: 
        dlstop=2555
    if dlstart>(dlstop-dlstep):
        dlstart=dlstop-dlstep
    dfp = datapath('DLScan')
    path, filename = os.path.split(dfp)
    result = str.format("File: {0}; Start: {1}; Stop: {2}; Step: {3}; Window: {4}; PreCount: {5}; HV: {6}",filename, dlstart, dlstop, dlstep, window, pre_cnt_window, hv)
    report(result, '.alpha.dlscan2')
    result = str.format("File: {0}; wScale: {1}; cntWin: {2}; winCorse: {3}; winFine: {4}",filename, wscale, cnt_window, window_corse, window_fine)
    report(result, '.alpha.dlscan2')
    print (filename)
    with open(dfp, 'a') as data:
        time_stamp = dt.now().strftime('%Y%m%d %H:%M:%S')
        data.write(f"dac\tcount1\tcount2\n")
        for i in range(dlstart, dlstop+dlstep, dlstep):
            await pmt1setdl( i/dacscale1)
            await pmt2setdl( i/dacscale1)
            try:
                await measurement_unit.ExecuteMeasurement(op_id)
                results = await measurement_unit.ReadMeasurementValues(op_id)
                #print(results)
                print(f"{i}\t{results[0]}\t{results[1]}")
                data.write(f"{i}\t{results[0]}\t{results[1]}\n")
            except BaseException as ex:
                print(f"darkcount() failed: {ex}")
                data.write(f"darkcount() failed: {ex}\n")
    await hvgate1off()
    await hvgate2off()
    await pmt1hvdisable()
    await pmt2hvdisable()
    report('Done', '.alpha.dlscan')
    print('Done')
    return(0)

async def hvscan(dl=64, hvstart=0, hvstop=188, hvstep=10, flashes=10, flash_mode=1, pre_cnt_window=10):
    '''
    ml 211027:
    hvscan, dual channel
    '''

    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit

    delay = 0.1
    dacscale1 = 256
    await pmt1setdl( dl/dacscale1)
    await pmt2setdl( dl/dacscale1)
    await pmt1sethv( 0)
    await pmt2sethv( 0)
    await pmt1hvenable()
    await pmt2hvenable()
    await hvgate1on()
    await hvgate2on()
    await pmt1sethv( hvstart/dacscale1)
    await pmt2sethv( hvstart/dacscale1)

    ## IntegratorModes
    full_reset = 0b010
    low_range_reset = 0b011
    integrate_autorange = 0b100
    integrate_with_fixed_range = 0b101
    integrate_in_low_range = 0b110
    integrate_in_high_range = 0b111

    lowresetdelay = 5000  # 50µs
    meastime = 5000       # 50µs
    delaytime = 5000      # 50µs
    integdelay = 500      # 5µs
    #pre_cnt_window = 100    # 1µs
    cnt_window = meastime // pre_cnt_window
    #window_corse, window_fine = divmod(cnt_window, 65536)

    power = 0.3
    frequency = 100
    #flash_mode = 1
    # Limit Freqency for High Power Mode
    if frequency > 500 and flash_mode == 1:
        frequency = 500

    # Set the Flash_Pwr
    await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampPower, power, timeout=1)

    # Set ontime based on Power Mode 
    if flash_mode == 1:
        ontime = 120000
        await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampHighPowerEnable, 1, timeout=1)
    else:
        ontime = 60000
        await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampHighPowerEnable, 0, timeout=1)

    offtime = int(1 / frequency * 1e8 - ontime)

    op_id = 'pmttest2a'
    seq_gen = meas_seq_generator()
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=0)   # pmt1, count_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=1)   # pmt1, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=2)   # pmt1, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=3)   # pmt1, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=4)   # pmt1, analog_high_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=5)   # pmt2, count_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=6)   # pmt2, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=7)   # pmt2, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=8)   # pmt2, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=9)   # pmt2, analog_high_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=10)  # ref, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=11)  # ref, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=12)  # ref, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=13)  # ref, analog_high_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=14)  # abs, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=15)  # abs, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=16)  # abs, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=17)  # abs, analog_high_offset
    seq_gen.Loop(flashes)
    seq_gen.TimerWaitAndRestart(ontime)
    seq_gen.SetSignals(OutputSignal.Flash)
    seq_gen.SetIntegratorMode(abs=full_reset, ref=full_reset, pmt2=full_reset, pmt1=full_reset)
    seq_gen.TimerWaitAndRestart(lowresetdelay)
    seq_gen.SetIntegratorMode(abs=low_range_reset, ref=low_range_reset, pmt2=low_range_reset, pmt1=low_range_reset)
    seq_gen.TimerWaitAndRestart(delaytime)
    # high range offset
    seq_gen.TimerWaitAndRestart(meastime)
    seq_gen.SetIntegratorMode(abs=integrate_in_high_range, ref=integrate_in_high_range, pmt2=integrate_in_high_range, pmt1=integrate_in_high_range)
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)
    # channels: pmt1= 0; pmt2= 1; ref= 4; abs= 5
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=4)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=9)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=13)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=17)
    # low range offset
    seq_gen.TimerWaitAndRestart(meastime)
    seq_gen.SetIntegratorMode(abs=integrate_in_low_range, ref=integrate_in_low_range, pmt2=integrate_in_low_range, pmt1=integrate_in_low_range)
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=3)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=8)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=12)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=16)
    seq_gen.ResetSignals(OutputSignal.Flash)
    # measurement
    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    seq_gen.SetIntegratorMode(abs=integrate_autorange, ref=integrate_autorange, pmt2=integrate_autorange, pmt1=integrate_autorange)
    seq_gen.PulseCounterControl(channel=0, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    seq_gen.PulseCounterControl(channel=1, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    seq_gen.Loop(cnt_window)
    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    seq_gen.PulseCounterControl(channel=0, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
    seq_gen.PulseCounterControl(channel=1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
    seq_gen.LoopEnd()
    seq_gen.TimerWaitAndRestart(integdelay)
    seq_gen.SetIntegratorMode(abs=integrate_with_fixed_range, ref=integrate_with_fixed_range, pmt2=integrate_with_fixed_range, pmt1=integrate_with_fixed_range)
    seq_gen.TimerWait()
    # read out analog signals
    seq_gen.GetPulseCounterResult(channel=0, relative=False, resetCounter=True, cumulative=True, dword=False, addrPos=0, resultPos=0)
    seq_gen.GetPulseCounterResult(channel=1, relative=False, resetCounter=True, cumulative=True, dword=False, addrPos=0, resultPos=5)
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=1)
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=2)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=6)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=7)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=10)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=11)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=14)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=15)
    seq_gen.TimerWaitAndRestart(offtime)
    seq_gen.LoopEnd()
    seq_gen.SetIntegratorMode(abs=full_reset, ref=full_reset, pmt2=full_reset, pmt1=full_reset)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    measurement_unit.resultAddresses[op_id] = range(0, 18)
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)

    if hvstop>hvmax: 
        hvstop=hvmax
    if hvstart>(hvstop-hvstep):
        hvstart=hvstop-hvstep
    dfp = datapath('HVScan')
    path, filename = os.path.split(dfp)
    result = str.format("File: {0}; DL: {1}; Start: {2}; Stop: {3}; Step: {4}; Flashes: {5}; Flash_Mode: {6}; PreCount: {7}; cntWin: {8}",filename, dl, hvstart, hvstop, hvstep, flashes, flash_mode, pre_cnt_window, cnt_window)
    report(result, '.alpha.hvscan')
    print (filename)
    with open(dfp, 'a') as data:
        time_stamp = dt.now().strftime('%Y%m%d %H:%M:%S')
        data.write(f"dac\tpmt1_cr\tpmt1_alr\tpmt1_ahr\tpmt1_alo\tpmt1_aho\tpmt2_cr\tpmt2_alr\tpmt2_ahr\tpmt2_alo\tpmt2_aho\tref_alr\tref_ahr\tref_alo\tref_aho\tabs_alr\tabs_ahr\tabs_alo\tabs_aho\n")
        for i in range(hvstart, hvstop+hvstep, hvstep):
            await pmt1sethv( i/dacscale1)
            await pmt2sethv( i/dacscale1)
            try:
                await measurement_unit.ExecuteMeasurement(op_id)
                results = await measurement_unit.ReadMeasurementValues(op_id)
                data.write(f"{i}\t{results[0]}\t{results[1]}\t{results[2]}\t{results[3]}\t{results[4]}\t{results[5]}\t{results[6]}\t{results[7]}\t{results[8]}\t{results[9]}\t{results[10]}\t{results[11]}\t{results[12]}\t{results[13]}\t{results[14]}\t{results[15]}\t{results[16]}\t{results[17]}\n")
            except BaseException as ex:
                print(f"darkcount() failed: {ex}")
                data.write(f"darkcount() failed: {ex}\n")
    await hvgate1off()
    await hvgate2off()
    await pmt1hvdisable()
    await pmt2hvdisable()
    await pmt1sethv(0)
    await pmt2sethv(0)
    report('Done', '.alpha.hvscan')
    print('Done')
    return(0)

async def darkcount(iterations, hv1 = 0.4, hv2 = 0.5, dl1 = 0.25, dl2 = 0.25, window=1000, pre_cnt_window=10000):
    '''
    ml 210906: darkcount measurement
    10ns timer tick
    1µs = 100 ticks
    1ms = 100000 ticks
    '''
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit

    delay = 0.1
    await pmt1setdl(dl1)
    await pmt2setdl(dl2)
    await pmt1sethv(hv1)
    await pmt2sethv(hv2)
    await pmt1hvenable()
    await pmt2hvenable()
    await hvgate1on()
    await hvgate2on()

    ## IntegratorModes
    full_reset = 0b010
    low_range_reset = 0b011
    integrate_autorange = 0b100
    integrate_with_fixed_range = 0b101
    integrate_in_low_range = 0b110
    integrate_in_high_range = 0b111

    highresetdelay = 5000 # 50µs
    lowresetdelay = 5000  # 50µs
    meastime = 5000       # 50µs
    delaytime = 5000      # 50µs
    integdelay = 500      # 5µs
    #pre_cnt_window = 100    # 1µs
    wscale = 100000 // pre_cnt_window   # 1ms scale for window
    cnt_window = window * wscale
    window_corse, window_fine = divmod(cnt_window, 65536)

    op_id = 'darktest'
    seq_gen = meas_seq_generator()
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=0)   # pmt1, count_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=1)   # pmt1, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=2)   # pmt1, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=3)   # pmt1, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=4)   # pmt1, analog_high_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=5)   # pmt2, count_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=6)   # pmt2, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=7)   # pmt2, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=8)   # pmt2, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=9)   # pmt2, analog_high_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=10)  # ref, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=11)  # ref, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=12)  # ref, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=13)  # ref, analog_high_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=14)  # abs, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=15)  # abs, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=16)  # abs, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=17)  # abs, analog_high_offset
    seq_gen.TimerWaitAndRestart(highresetdelay)
    seq_gen.SetSignals(OutputSignal.Flash)
    seq_gen.SetIntegratorMode(abs=full_reset, ref=full_reset, pmt2=full_reset, pmt1=full_reset)
    seq_gen.TimerWaitAndRestart(lowresetdelay)
    seq_gen.SetIntegratorMode(abs=low_range_reset, ref=low_range_reset, pmt2=low_range_reset, pmt1=low_range_reset)
    seq_gen.TimerWaitAndRestart(delaytime)
    # high range offset
    seq_gen.TimerWaitAndRestart(meastime)
    seq_gen.SetIntegratorMode(abs=integrate_in_high_range, ref=integrate_in_high_range, pmt2=integrate_in_high_range, pmt1=integrate_in_high_range)
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)
    # channels: pmt1= 0; pmt2= 1; ref= 4; abs= 5
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=4)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=9)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=13)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=17)
    # low range offset
    seq_gen.TimerWaitAndRestart(meastime)
    seq_gen.SetIntegratorMode(abs=integrate_in_low_range, ref=integrate_in_low_range, pmt2=integrate_in_low_range, pmt1=integrate_in_low_range)
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=3)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=8)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=12)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=16)
    # measurement
    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    seq_gen.SetIntegratorMode(abs=integrate_autorange, ref=integrate_autorange, pmt2=integrate_autorange, pmt1=integrate_autorange)
    seq_gen.PulseCounterControl(channel=0, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    seq_gen.PulseCounterControl(channel=1, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    if (window_corse > 0):
        seq_gen.Loop(window_corse)
        seq_gen.Loop(65536)
        seq_gen.TimerWaitAndRestart(pre_cnt_window)
        seq_gen.PulseCounterControl(channel=0, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
        seq_gen.PulseCounterControl(channel=1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
        seq_gen.LoopEnd()
        seq_gen.LoopEnd()
    if (window_fine > 0):
        seq_gen.Loop(window_fine)
        seq_gen.TimerWaitAndRestart(pre_cnt_window)
        seq_gen.PulseCounterControl(channel=0, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
        seq_gen.PulseCounterControl(channel=1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
        seq_gen.LoopEnd()
    seq_gen.TimerWaitAndRestart(integdelay)
    seq_gen.SetIntegratorMode(abs=integrate_with_fixed_range, ref=integrate_with_fixed_range, pmt2=integrate_with_fixed_range, pmt1=integrate_with_fixed_range)
    seq_gen.TimerWait()
    # read out analog signals
    seq_gen.GetPulseCounterResult(channel=0, relative=False, resetCounter=True, cumulative=True, dword=False, addrPos=0, resultPos=0)
    seq_gen.GetPulseCounterResult(channel=1, relative=False, resetCounter=True, cumulative=True, dword=False, addrPos=0, resultPos=5)
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=1)
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=2)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=6)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=7)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=10)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=11)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=14)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=15)
    seq_gen.SetIntegratorMode(abs=full_reset, ref=full_reset, pmt2=full_reset, pmt1=full_reset)
    seq_gen.Stop(0)

    measurement_unit.ClearOperations()
    measurement_unit.resultAddresses[op_id] = range(0, 18)
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)


    result = str.format("iterations: {0}, Count-Window: {1}, PreCount-Window: {2}",iterations, cnt_window, pre_cnt_window)
    report(result, '.alpha.darkcount1')
    dfp = datapath('DarkCount')
    path, filename = os.path.split(dfp)
    report(filename, '.alpha.darkcount1'); #Write to Report
    print (filename)
    with open(dfp, 'a') as data:
        data.write(f"iter\tpmt1_cr\tpmt1_alr\tpmt1_ahr\tpmt1_alo\tpmt1_aho\tpmt2_cr\tpmt2_alr\tpmt2_ahr\tpmt2_alo\tpmt2_aho\tref_alr\tref_ahr\tref_alo\tref_aho\tabs_alr\tabs_ahr\tabs_alo\tabs_aho\tTime\n")
        for i in range(iterations):
            try:
                time_stamp1 = dt.now().strftime('%H:%M:%S.%f')                
                await measurement_unit.ExecuteMeasurement(op_id)
                await asyncio.sleep(delay)
                results = await measurement_unit.ReadMeasurementValues(op_id)
                print(f"{i}\t{results[0]}\t{results[5]}")
                data.write(f"{i}\t{results[0]}\t{results[1]}\t{results[2]}\t{results[3]}\t{results[4]}\t{results[5]}\t{results[6]}\t{results[7]}\t{results[8]}\t{results[9]}\t{results[10]}\t{results[11]}\t{results[12]}\t{results[13]}\t{results[14]}\t{results[15]}\t{results[16]}\t{results[17]}\t{time_stamp1}\n")
            except BaseException as ex:
                print(f"darkcount() failed: {ex}")
                data.write(f"darkcount() failed: {ex}\n")
    await hvgate1off()
    await hvgate2off()
    await pmt1hvdisable()
    await pmt2hvdisable()
    return(0)

async def pmttest1a(iterations, flashes=1, flash_mode=1, frequency=100, hv1 = 0.4, hv2 = 0.5, dl1 = 0.25, dl2 = 0.25, window=1000, pre_cnt_window=100):
    '''
    ml 211027: pmttest measurement
	analog only
    10ns timer tick
    1µs = 100 ticks
    1ms = 100000 ticks
    '''
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit

    #pre_cnt_window = 100                # 1µs
    wscale = 100000 // pre_cnt_window   # 1ms scale for window
    cnt_window = window * wscale
    window_corse, window_fine = divmod(cnt_window, 65536)

    delay = 0.1
    await pmt1setdl(dl1)
    await pmt2setdl(dl2)
    await pmt1sethv(hv1)
    await pmt2sethv(hv2)
    await pmt1hvenable()
    await pmt2hvenable()
    await hvgate1on()
    await hvgate2on()

    ## IntegratorModes
    full_reset = 0b010
    low_range_reset = 0b011
    integrate_autorange = 0b100
    integrate_with_fixed_range = 0b101
    integrate_in_low_range = 0b110
    integrate_in_high_range = 0b111

    lowresetdelay = 5000  # 50µs
    meastime = 5000       # 50µs
    delaytime = 5000      # 50µs
    integdelay = 500      # 5µs
    power = 0.8
    # Limit Freqency for High Power Mode
    if frequency > 500 and flash_mode == 1:
        frequency = 500

    # Set the Flash_Pwr
    await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampPower, power, timeout=1)

    # Set ontime based on Power Mode 
    if flash_mode == 1:
        ontime = 120000
        await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampHighPowerEnable, 0, timeout=1)
    else:
        ontime = 60000
        await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampHighPowerEnable, 1, timeout=1)

    offtime = int(1 / frequency * 1e8 - ontime)

    dfp = datapath('PMTTest')
    path, filename = os.path.split(dfp)
    result = str.format("Filename: {0}; Flashes: {1}; Flash Mode: {2}; Flash-Power: {3}; Iterations: {4}",filename, flashes, flash_mode, power, iterations)
    report(result, '.alpha.pmttest1a')
    print (filename)

    op_id = 'pmttest1a'
    seq_gen = meas_seq_generator()
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=0)   # pmt1, count_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=1)   # pmt1, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=2)   # pmt1, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=3)   # pmt1, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=4)   # pmt1, analog_high_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=5)   # pmt2, count_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=6)   # pmt2, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=7)   # pmt2, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=8)   # pmt2, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=9)   # pmt2, analog_high_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=10)  # ref, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=11)  # ref, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=12)  # ref, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=13)  # ref, analog_high_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=14)  # abs, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=15)  # abs, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=16)  # abs, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=17)  # abs, analog_high_offset
    seq_gen.Loop(flashes)
    seq_gen.TimerWaitAndRestart(ontime)
    seq_gen.SetSignals(OutputSignal.Flash)
    seq_gen.SetIntegratorMode(abs=full_reset, ref=full_reset, pmt2=full_reset, pmt1=full_reset)
    seq_gen.TimerWaitAndRestart(lowresetdelay)
    seq_gen.SetIntegratorMode(abs=low_range_reset, ref=low_range_reset, pmt2=low_range_reset, pmt1=low_range_reset)
    seq_gen.TimerWaitAndRestart(delaytime)
    # high range offset
    seq_gen.TimerWaitAndRestart(meastime)
    seq_gen.SetIntegratorMode(abs=integrate_in_high_range, ref=integrate_in_high_range, pmt2=integrate_in_high_range, pmt1=integrate_in_high_range)
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)
    # channels: pmt1= 0; pmt2= 1; ref= 4; abs= 5
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=4)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=9)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=13)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=17)
    # low range offset
    seq_gen.TimerWaitAndRestart(meastime)
    seq_gen.SetIntegratorMode(abs=integrate_in_low_range, ref=integrate_in_low_range, pmt2=integrate_in_low_range, pmt1=integrate_in_low_range)
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=3)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=8)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=12)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=16)
    seq_gen.ResetSignals(OutputSignal.Flash)
    # measurement
    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    seq_gen.SetIntegratorMode(abs=integrate_autorange, ref=integrate_autorange, pmt2=integrate_autorange, pmt1=integrate_autorange)
    seq_gen.PulseCounterControl(channel=0, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    seq_gen.PulseCounterControl(channel=1, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    if (window_corse > 0):
        seq_gen.Loop(window_corse)
        seq_gen.Loop(65536)
        seq_gen.TimerWaitAndRestart(pre_cnt_window)
        seq_gen.PulseCounterControl(channel=0, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
        seq_gen.PulseCounterControl(channel=1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
        seq_gen.LoopEnd()
        seq_gen.LoopEnd()
    if (window_fine > 0):
        seq_gen.Loop(window_fine)
        seq_gen.TimerWaitAndRestart(pre_cnt_window)
        seq_gen.PulseCounterControl(channel=0, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
        seq_gen.PulseCounterControl(channel=1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
        seq_gen.LoopEnd()
    seq_gen.TimerWaitAndRestart(integdelay)
    seq_gen.SetIntegratorMode(abs=integrate_with_fixed_range, ref=integrate_with_fixed_range, pmt2=integrate_with_fixed_range, pmt1=integrate_with_fixed_range)
    seq_gen.TimerWait()
    # read out analog signals
    seq_gen.GetPulseCounterResult(channel=0, relative=False, resetCounter=True, cumulative=True, dword=False, addrPos=0, resultPos=0)
    seq_gen.GetPulseCounterResult(channel=1, relative=False, resetCounter=True, cumulative=True, dword=False, addrPos=0, resultPos=5)
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=1)
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=2)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=6)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=7)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=10)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=11)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=14)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=15)
    seq_gen.TimerWaitAndRestart(offtime)
    seq_gen.LoopEnd()
    seq_gen.SetIntegratorMode(abs=full_reset, ref=full_reset, pmt2=full_reset, pmt1=full_reset)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    measurement_unit.resultAddresses[op_id] = range(0, 18)
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)

    with open(dfp, 'a') as data:
        data.write(f"iter\tpmt1_cr\tpmt1_alr\tpmt1_ahr\tpmt1_alo\tpmt1_aho\tpmt2_cr\tpmt2_alr\tpmt2_ahr\tpmt2_alo\tpmt2_aho\tref_alr\tref_ahr\tref_alo\tref_aho\tabs_alr\tabs_ahr\tabs_alo\tabs_aho\n")
        for i in range(iterations):
            try:
                await measurement_unit.ExecuteMeasurement(op_id)
                await asyncio.sleep(delay)
                results = await measurement_unit.ReadMeasurementValues(op_id)
                data.write(f"{i}\t{results[0]}\t{results[1]}\t{results[2]}\t{results[3]}\t{results[4]}\t{results[5]}\t{results[6]}\t{results[7]}\t{results[8]}\t{results[9]}\t{results[10]}\t{results[11]}\t{results[12]}\t{results[13]}\t{results[14]}\t{results[15]}\t{results[16]}\t{results[17]}\n")
            except BaseException as ex:
                print(f"pmttest1a() failed: {ex}")
            await asyncio.sleep(delay)

    report('Done', '.alpha.pmttest1a')
    print('pmttest1a Done')
    return (0)

async def pmttest1b(iterations, flashes=1, flash_mode=1, frequency=100, hv1 = 0.4, hv2 = 0.5, dl1 = 0.25, dl2 = 0.25, window=1000, pre_cnt_window=100):
    '''
    ml 211027: pmttest measurement
	analog only
    10ns timer tick
    1µs = 100 ticks
    1ms = 100000 ticks
    '''
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit

    #pre_cnt_window = 100                # 1µs
    wscale = 100000 // pre_cnt_window   # 1ms scale for window
    cnt_window = window * wscale
    window_corse, window_fine = divmod(cnt_window, 65536)

    delay = 0.1
    await pmt1setdl(dl1)
    await pmt2setdl(dl2)
    await pmt1sethv(hv1)
    await pmt2sethv(hv2)
    await pmt1hvenable()
    await pmt2hvenable()
    await hvgate1on()
    await hvgate2on()

    ## IntegratorModes
    full_reset = 0b010
    low_range_reset = 0b011
    integrate_autorange = 0b100
    integrate_with_fixed_range = 0b101
    integrate_in_low_range = 0b110
    integrate_in_high_range = 0b111

    lowresetdelay = 5000  # 50µs
    meastime = 5000       # 50µs
    delaytime = 5000      # 50µs
    integdelay = 500      # 5µs
    power = 3277          # 0.8
    # Limit Freqency for High Power Mode
    if frequency > 500 and flash_mode == 1:
        frequency = 500

    # Set the Flash_Pwr
    await flashsetpower(power)

    # Set ontime based on Power Mode 
    if flash_mode == 1:
        ontime = 120000
        await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampHighPowerEnable, 0, timeout=1)
    else:
        ontime = 60000
        await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampHighPowerEnable, 1, timeout=1)

    offtime = int(1 / frequency * 1e8 - ontime)

    dfp = datapath('PMTTest')
    path, filename = os.path.split(dfp)
    result = str.format("Filename: {0}; Flashes: {1}; Flash Mode: {2}; Flash-Power: {3}; Iterations: {4}",filename, flashes, flash_mode, power, iterations)
    report(result, '.alpha.pmttest1b')
    print (filename)

    op_id = 'pmttest1b'
    seq_gen = meas_seq_generator()
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=0)   # pmt1, count_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=1)   # pmt1, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=2)   # pmt1, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=3)   # pmt1, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=4)   # pmt1, analog_high_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=5)   # pmt2, count_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=6)   # pmt2, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=7)   # pmt2, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=8)   # pmt2, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=9)   # pmt2, analog_high_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=10)  # ref, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=11)  # ref, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=12)  # ref, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=13)  # ref, analog_high_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=14)  # abs, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=15)  # abs, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=16)  # abs, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=17)  # abs, analog_high_offset
    seq_gen.Loop(flashes)
    seq_gen.TimerWaitAndRestart(ontime)
    #seq_gen.SetSignals(OutputSignal.Flash)
    seq_gen.SetIntegratorMode(abs=full_reset, ref=full_reset, pmt2=full_reset, pmt1=full_reset)
    seq_gen.TimerWaitAndRestart(lowresetdelay)
    seq_gen.SetIntegratorMode(abs=low_range_reset, ref=low_range_reset, pmt2=low_range_reset, pmt1=low_range_reset)
    seq_gen.TimerWaitAndRestart(delaytime)
    # high range offset
    seq_gen.TimerWaitAndRestart(meastime)
    seq_gen.SetIntegratorMode(abs=integrate_in_high_range, ref=integrate_in_high_range, pmt2=integrate_in_high_range, pmt1=integrate_in_high_range)
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)
    # channels: pmt1= 0; pmt2= 1; ref= 4; abs= 5
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=4)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=9)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=13)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=17)
    # low range offset
    seq_gen.TimerWaitAndRestart(meastime)
    seq_gen.SetIntegratorMode(abs=integrate_in_low_range, ref=integrate_in_low_range, pmt2=integrate_in_low_range, pmt1=integrate_in_low_range)
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=3)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=8)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=12)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=16)
    #seq_gen.ResetSignals(OutputSignal.Flash)
    # measurement
    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    seq_gen.SetIntegratorMode(abs=integrate_autorange, ref=integrate_autorange, pmt2=integrate_autorange, pmt1=integrate_autorange)
    seq_gen.PulseCounterControl(channel=0, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    seq_gen.PulseCounterControl(channel=1, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    if (window_corse > 0):
        seq_gen.Loop(window_corse)
        seq_gen.Loop(65536)
        seq_gen.TimerWaitAndRestart(pre_cnt_window)
        seq_gen.PulseCounterControl(channel=0, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
        seq_gen.PulseCounterControl(channel=1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
        seq_gen.LoopEnd()
        seq_gen.LoopEnd()
    if (window_fine > 0):
        seq_gen.Loop(window_fine)
        seq_gen.TimerWaitAndRestart(pre_cnt_window)
        seq_gen.PulseCounterControl(channel=0, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
        seq_gen.PulseCounterControl(channel=1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
        seq_gen.LoopEnd()
    seq_gen.TimerWaitAndRestart(integdelay)
    seq_gen.SetIntegratorMode(abs=integrate_with_fixed_range, ref=integrate_with_fixed_range, pmt2=integrate_with_fixed_range, pmt1=integrate_with_fixed_range)
    seq_gen.TimerWait()
    # read out analog signals
    seq_gen.GetPulseCounterResult(channel=0, relative=False, resetCounter=True, cumulative=True, dword=False, addrPos=0, resultPos=0)
    seq_gen.GetPulseCounterResult(channel=1, relative=False, resetCounter=True, cumulative=True, dword=False, addrPos=0, resultPos=5)
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=1)
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=2)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=6)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=7)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=10)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=11)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=14)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=15)
    seq_gen.TimerWaitAndRestart(offtime)
    seq_gen.LoopEnd()
    seq_gen.SetIntegratorMode(abs=full_reset, ref=full_reset, pmt2=full_reset, pmt1=full_reset)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    measurement_unit.resultAddresses[op_id] = range(0, 18)
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)

    with open(dfp, 'a') as data:
        data.write(f"iter\tpmt1_cr\tpmt1_alr\tpmt1_ahr\tpmt1_alo\tpmt1_aho\tpmt2_cr\tpmt2_alr\tpmt2_ahr\tpmt2_alo\tpmt2_aho\tref_alr\tref_ahr\tref_alo\tref_aho\tabs_alr\tabs_ahr\tabs_alo\tabs_aho\n")
        for i in range(iterations):
            try:
                await measurement_unit.ExecuteMeasurement(op_id)
                await asyncio.sleep(delay)
                results = await measurement_unit.ReadMeasurementValues(op_id)
                data.write(f"{i}\t{results[0]}\t{results[1]}\t{results[2]}\t{results[3]}\t{results[4]}\t{results[5]}\t{results[6]}\t{results[7]}\t{results[8]}\t{results[9]}\t{results[10]}\t{results[11]}\t{results[12]}\t{results[13]}\t{results[14]}\t{results[15]}\t{results[16]}\t{results[17]}\n")
            except BaseException as ex:
                print(f"pmttest1b() failed: {ex}")
            await asyncio.sleep(delay)

    report('Done', '.alpha.pmttest1b')
    print('pmttest1b Done')
    return (0)

async def pmttest2a(iterations, flashes=1, flash_mode=1, frequency=100, hv1 = 0.4, hv2 = 0.5, dl1 = 0.25, dl2 = 0.25, pre_cnt_window=10):
    '''
    ml 211027: pmttest measurement
	50µs Window, with Flash
    10ns timer tick
    1µs = 100 ticks
    1ms = 100000 ticks
    '''
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit

    delay = 0.1
    await pmt1setdl(dl1)
    await pmt2setdl(dl2)
    await pmt1sethv(hv1)
    await pmt2sethv(hv2)
    await pmt1hvenable()
    await pmt2hvenable()
    await hvgate1on()
    await hvgate2on()

    ## IntegratorModes
    full_reset = 0b010
    low_range_reset = 0b011
    integrate_autorange = 0b100
    integrate_with_fixed_range = 0b101
    integrate_in_low_range = 0b110
    integrate_in_high_range = 0b111

    lowresetdelay = 5000  # 50µs
    meastime = 5000       # 50µs
    delaytime = 5000      # 50µs
    integdelay = 500      # 5µs
    #pre_cnt_window = 100    # 1µs
    cnt_window = meastime // pre_cnt_window
    #window_corse, window_fine = divmod(cnt_window, 65536)

    power = 0.8
    # Limit Freqency for High Power Mode
    if frequency > 500 and flash_mode == 1:
        frequency = 500

    # Set the Flash_Pwr
    await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampPower, power, timeout=1)

    # Set ontime based on Power Mode 
    if flash_mode == 1:
        ontime = 120000
        await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampHighPowerEnable, 0, timeout=1)
    else:
        ontime = 60000
        await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampHighPowerEnable, 1, timeout=1)

    offtime = int(1 / frequency * 1e8 - ontime)

    dfp = datapath('PMTTest')
    path, filename = os.path.split(dfp)
    result = str.format("Filename: {0}; Flashes: {1}; Flash Mode: {2}; Flash-Power: {3}; Iterations: {4}; Frequency: {5}",filename, flashes, flash_mode, power, iterations, frequency)
    report(result, '.alpha.pmttest2a')
    result = str.format("HV1: {0}; HV2: {1}; DL1: {2}; DL2: {3}; CntWindow: {4};PreCntWindow: {5}", hv1, hv2, dl1, dl2, cnt_window, pre_cnt_window)
    report(result, '.alpha.pmttest2a')
    print (filename)

    op_id = 'pmttest2a'
    seq_gen = meas_seq_generator()
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=0)   # pmt1, count_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=1)   # pmt1, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=2)   # pmt1, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=3)   # pmt1, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=4)   # pmt1, analog_high_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=5)   # pmt2, count_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=6)   # pmt2, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=7)   # pmt2, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=8)   # pmt2, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=9)   # pmt2, analog_high_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=10)  # ref, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=11)  # ref, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=12)  # ref, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=13)  # ref, analog_high_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=14)  # abs, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=15)  # abs, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=16)  # abs, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=17)  # abs, analog_high_offset
    seq_gen.Loop(flashes)
    seq_gen.TimerWaitAndRestart(ontime)
    seq_gen.SetSignals(OutputSignal.Flash)
    seq_gen.SetIntegratorMode(abs=full_reset, ref=full_reset, pmt2=full_reset, pmt1=full_reset)
    seq_gen.TimerWaitAndRestart(lowresetdelay)
    seq_gen.SetIntegratorMode(abs=low_range_reset, ref=low_range_reset, pmt2=low_range_reset, pmt1=low_range_reset)
    seq_gen.TimerWaitAndRestart(delaytime)
    # high range offset
    seq_gen.TimerWaitAndRestart(meastime)
    seq_gen.SetIntegratorMode(abs=integrate_in_high_range, ref=integrate_in_high_range, pmt2=integrate_in_high_range, pmt1=integrate_in_high_range)
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)
    # channels: pmt1= 0; pmt2= 1; ref= 4; abs= 5
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=4)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=9)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=13)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=17)
    # low range offset
    seq_gen.TimerWaitAndRestart(meastime)
    seq_gen.SetIntegratorMode(abs=integrate_in_low_range, ref=integrate_in_low_range, pmt2=integrate_in_low_range, pmt1=integrate_in_low_range)
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=3)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=8)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=12)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=16)
    seq_gen.ResetSignals(OutputSignal.Flash)
    # measurement
    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    seq_gen.SetIntegratorMode(abs=integrate_autorange, ref=integrate_autorange, pmt2=integrate_autorange, pmt1=integrate_autorange)
    seq_gen.PulseCounterControl(channel=0, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    seq_gen.PulseCounterControl(channel=1, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    seq_gen.Loop(cnt_window)
    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    seq_gen.PulseCounterControl(channel=0, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
    seq_gen.PulseCounterControl(channel=1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
    seq_gen.LoopEnd()
    seq_gen.TimerWaitAndRestart(integdelay)
    seq_gen.SetIntegratorMode(abs=integrate_with_fixed_range, ref=integrate_with_fixed_range, pmt2=integrate_with_fixed_range, pmt1=integrate_with_fixed_range)
    seq_gen.TimerWait()
    # read out analog signals
    seq_gen.GetPulseCounterResult(channel=0, relative=False, resetCounter=True, cumulative=True, dword=False, addrPos=0, resultPos=0)
    seq_gen.GetPulseCounterResult(channel=1, relative=False, resetCounter=True, cumulative=True, dword=False, addrPos=0, resultPos=5)
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=1)
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=2)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=6)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=7)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=10)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=11)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=14)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=15)
    seq_gen.TimerWaitAndRestart(offtime)
    seq_gen.LoopEnd()
    seq_gen.SetIntegratorMode(abs=full_reset, ref=full_reset, pmt2=full_reset, pmt1=full_reset)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    measurement_unit.resultAddresses[op_id] = range(0, 18)
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)

    with open(dfp, 'a') as data:
        data.write(f"iter\tpmt1_cr\tpmt1_alr\tpmt1_ahr\tpmt1_alo\tpmt1_aho\tpmt2_cr\tpmt2_alr\tpmt2_ahr\tpmt2_alo\tpmt2_aho\tref_alr\tref_ahr\tref_alo\tref_aho\tabs_alr\tabs_ahr\tabs_alo\tabs_aho\n")
        for i in range(iterations):
            try:
                await measurement_unit.ExecuteMeasurement(op_id)
                await asyncio.sleep(delay)
                results = await measurement_unit.ReadMeasurementValues(op_id)
                data.write(f"{i}\t{results[0]}\t{results[1]}\t{results[2]}\t{results[3]}\t{results[4]}\t{results[5]}\t{results[6]}\t{results[7]}\t{results[8]}\t{results[9]}\t{results[10]}\t{results[11]}\t{results[12]}\t{results[13]}\t{results[14]}\t{results[15]}\t{results[16]}\t{results[17]}\n")
            except BaseException as ex:
                print(f"pmttest2a() failed: {ex}")
            await asyncio.sleep(delay)

    report('Done', '.alpha.pmttest2a')
    print('pmttest2a Done')
    return (0)

async def pmttest2b(iterations, flashes=1, flash_mode=1, frequency=100, hv1 = 0.4, hv2 = 0.5, dl1 = 0.25, dl2 = 0.25, pre_cnt_window=10):
    '''
    ml 211027: pmttest measurement
	50µs Window, without Flash
    10ns timer tick
    1µs = 100 ticks
    1ms = 100000 ticks
    '''
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit

    delay = 0.1
    await pmt1setdl(dl1)
    await pmt2setdl(dl2)
    await pmt1sethv(hv1)
    await pmt2sethv(hv2)
    await pmt1hvenable()
    await pmt2hvenable()
    await hvgate1on()
    await hvgate2on()

    ## IntegratorModes
    full_reset = 0b010
    low_range_reset = 0b011
    integrate_autorange = 0b100
    integrate_with_fixed_range = 0b101
    integrate_in_low_range = 0b110
    integrate_in_high_range = 0b111

    lowresetdelay = 5000  # 50µs
    meastime = 5000       # 50µs
    delaytime = 5000      # 50µs
    integdelay = 500      # 5µs
    #pre_cnt_window = 100    # 1µs
    cnt_window = meastime // pre_cnt_window
    #window_corse, window_fine = divmod(cnt_window, 65536)

    power = 0.8
    # Limit Freqency for High Power Mode
    if frequency > 500 and flash_mode == 1:
        frequency = 500

    # Set the Flash_Pwr
    await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampPower, power, timeout=1)

    # Set ontime based on Power Mode 
    if flash_mode == 1:
        ontime = 120000
        await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampHighPowerEnable, 0, timeout=1)
    else:
        ontime = 60000
        await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampHighPowerEnable, 1, timeout=1)

    offtime = int(1 / frequency * 1e8 - ontime)

    dfp = datapath('PMTTest')
    path, filename = os.path.split(dfp)
    result = str.format("Filename: {0}; Flashes: {1}; Flash Mode: {2}; Flash-Power: {3}; Iterations: {4}; Frequency: {5}",filename, flashes, flash_mode, power, iterations, frequency)
    report(result, '.alpha.pmttest2b')
    result = str.format("HV1: {0}; HV2: {1}; DL1: {2}; DL2: {3}; CntWindow: {4};PreCntWindow: {5}", hv1, hv2, dl1, dl2, cnt_window, pre_cnt_window)
    report(result, '.alpha.pmttest2b')
    print (filename)

    op_id = 'pmttest2b'
    seq_gen = meas_seq_generator()
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=0)   # pmt1, count_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=1)   # pmt1, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=2)   # pmt1, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=3)   # pmt1, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=4)   # pmt1, analog_high_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=5)   # pmt2, count_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=6)   # pmt2, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=7)   # pmt2, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=8)   # pmt2, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=9)   # pmt2, analog_high_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=10)  # ref, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=11)  # ref, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=12)  # ref, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=13)  # ref, analog_high_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=14)  # abs, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=15)  # abs, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=16)  # abs, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=17)  # abs, analog_high_offset
    seq_gen.Loop(flashes)
    seq_gen.TimerWaitAndRestart(ontime)
    #seq_gen.SetSignals(OutputSignal.Flash)
    seq_gen.SetIntegratorMode(abs=full_reset, ref=full_reset, pmt2=full_reset, pmt1=full_reset)
    seq_gen.TimerWaitAndRestart(lowresetdelay)
    seq_gen.SetIntegratorMode(abs=low_range_reset, ref=low_range_reset, pmt2=low_range_reset, pmt1=low_range_reset)
    seq_gen.TimerWaitAndRestart(delaytime)
    # high range offset
    seq_gen.TimerWaitAndRestart(meastime)
    seq_gen.SetIntegratorMode(abs=integrate_in_high_range, ref=integrate_in_high_range, pmt2=integrate_in_high_range, pmt1=integrate_in_high_range)
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)
    # channels: pmt1= 0; pmt2= 1; ref= 4; abs= 5
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=4)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=9)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=13)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=17)
    # low range offset
    seq_gen.TimerWaitAndRestart(meastime)
    seq_gen.SetIntegratorMode(abs=integrate_in_low_range, ref=integrate_in_low_range, pmt2=integrate_in_low_range, pmt1=integrate_in_low_range)
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=3)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=8)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=12)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=16)
    #seq_gen.ResetSignals(OutputSignal.Flash)
    # measurement
    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    seq_gen.SetIntegratorMode(abs=integrate_autorange, ref=integrate_autorange, pmt2=integrate_autorange, pmt1=integrate_autorange)
    seq_gen.PulseCounterControl(channel=0, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    seq_gen.PulseCounterControl(channel=1, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    seq_gen.Loop(cnt_window)
    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    seq_gen.PulseCounterControl(channel=0, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
    seq_gen.PulseCounterControl(channel=1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
    seq_gen.LoopEnd()
    seq_gen.TimerWaitAndRestart(integdelay)
    seq_gen.SetIntegratorMode(abs=integrate_with_fixed_range, ref=integrate_with_fixed_range, pmt2=integrate_with_fixed_range, pmt1=integrate_with_fixed_range)
    seq_gen.TimerWait()
    # read out analog signals
    seq_gen.GetPulseCounterResult(channel=0, relative=False, resetCounter=True, cumulative=True, dword=False, addrPos=0, resultPos=0)
    seq_gen.GetPulseCounterResult(channel=1, relative=False, resetCounter=True, cumulative=True, dword=False, addrPos=0, resultPos=5)
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=1)
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=2)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=6)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=7)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=10)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=11)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=14)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=15)
    seq_gen.TimerWaitAndRestart(offtime)
    seq_gen.LoopEnd()
    seq_gen.SetIntegratorMode(abs=full_reset, ref=full_reset, pmt2=full_reset, pmt1=full_reset)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    measurement_unit.resultAddresses[op_id] = range(0, 18)
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)

    with open(dfp, 'a') as data:
        data.write(f"iter\tpmt1_cr\tpmt1_alr\tpmt1_ahr\tpmt1_alo\tpmt1_aho\tpmt2_cr\tpmt2_alr\tpmt2_ahr\tpmt2_alo\tpmt2_aho\tref_alr\tref_ahr\tref_alo\tref_aho\tabs_alr\tabs_ahr\tabs_alo\tabs_aho\n")
        for i in range(iterations):
            try:
                await measurement_unit.ExecuteMeasurement(op_id)
                await asyncio.sleep(delay)
                results = await measurement_unit.ReadMeasurementValues(op_id)
                data.write(f"{i}\t{results[0]}\t{results[1]}\t{results[2]}\t{results[3]}\t{results[4]}\t{results[5]}\t{results[6]}\t{results[7]}\t{results[8]}\t{results[9]}\t{results[10]}\t{results[11]}\t{results[12]}\t{results[13]}\t{results[14]}\t{results[15]}\t{results[16]}\t{results[17]}\n")
            except BaseException as ex:
                print(f"pmttest2b() failed: {ex}")
            await asyncio.sleep(delay)

    report('Done', '.alpha.pmttest2b')
    print('pmttest2b Done')
    return (0)

async def pmttest3a(iterations, flashes=1, flash_mode=1, frequency=100, hv1 = 0.4, hv2 = 0.5, dl1 = 0.25, dl2 = 0.25, pre_cnt_window=10):
    '''
    ml 211027: pmttest measurement
	50µs Window, with Flash
    10ns timer tick
    1µs = 100 ticks
    1ms = 100000 ticks
    '''
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit

    delay = 0.1
    await pmt1setdl(dl1)
    await pmt2setdl(dl2)
    await pmt1sethv(hv1)
    await pmt2sethv(hv2)
    await pmt1hvenable()
    await pmt2hvenable()
    await hvgate1on()
    await hvgate2on()

    ## IntegratorModes
    full_reset = 0b010
    low_range_reset = 0b011
    integrate_autorange = 0b100
    integrate_with_fixed_range = 0b101
    integrate_in_low_range = 0b110
    integrate_in_high_range = 0b111

    lowresetdelay = 5000  # 50µs
    meastime = 5000       # 50µs
    delaytime = 5000      # 50µs
    integdelay = 500      # 5µs
    #pre_cnt_window = 100    # 1µs
    cnt_window = meastime // pre_cnt_window
    #window_corse, window_fine = divmod(cnt_window, 65536)

    power = 0.8
    # Limit Freqency for High Power Mode
    if frequency > 500 and flash_mode == 1:
        frequency = 500

    # Set the Flash_Pwr
    await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampPower, power, timeout=1)

    # Set ontime based on Power Mode 
    if flash_mode == 1:
        ontime = 120000
        await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampHighPowerEnable, 1, timeout=1)
    else:
        ontime = 60000
        await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampHighPowerEnable, 0, timeout=1)

    offtime = int(1 / frequency * 1e8 - ontime)

    op_id = 'pmttest3a'
    seq_gen = meas_seq_generator()
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=0)   # pmt1, count_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=1)   # pmt1, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=2)   # pmt1, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=3)   # pmt1, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=4)   # pmt1, analog_high_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=5)   # pmt2, count_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=6)   # pmt2, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=7)   # pmt2, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=8)   # pmt2, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=9)   # pmt2, analog_high_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=10)  # ref, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=11)  # ref, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=12)  # ref, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=13)  # ref, analog_high_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=14)  # abs, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=15)  # abs, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=16)  # abs, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=17)  # abs, analog_high_offset
    seq_gen.Loop(flashes)
    seq_gen.TimerWaitAndRestart(ontime)
    seq_gen.SetSignals(OutputSignal.Flash)
    seq_gen.SetIntegratorMode(abs=full_reset, ref=full_reset, pmt2=full_reset, pmt1=full_reset)
    seq_gen.TimerWaitAndRestart(lowresetdelay)
    seq_gen.SetIntegratorMode(abs=low_range_reset, ref=low_range_reset, pmt2=low_range_reset, pmt1=low_range_reset)
    seq_gen.TimerWaitAndRestart(delaytime)
    # high range offset
    seq_gen.TimerWaitAndRestart(meastime)
    seq_gen.SetIntegratorMode(abs=integrate_in_high_range, ref=integrate_in_high_range, pmt2=integrate_in_high_range, pmt1=integrate_in_high_range)
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)
    # channels: pmt1= 0; pmt2= 1; ref= 4; abs= 5
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=4)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=9)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=13)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=17)
    # low range offset
    seq_gen.TimerWaitAndRestart(meastime)
    seq_gen.SetIntegratorMode(abs=integrate_in_low_range, ref=integrate_in_low_range, pmt2=integrate_in_low_range, pmt1=integrate_in_low_range)
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=3)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=8)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=12)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=16)
    seq_gen.ResetSignals(OutputSignal.Flash)
    # measurement
    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    seq_gen.SetIntegratorMode(abs=integrate_autorange, ref=integrate_autorange, pmt2=integrate_autorange, pmt1=integrate_autorange)
    seq_gen.PulseCounterControl(channel=0, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    seq_gen.PulseCounterControl(channel=1, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    seq_gen.Loop(cnt_window)
    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    seq_gen.PulseCounterControl(channel=0, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
    seq_gen.PulseCounterControl(channel=1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
    seq_gen.LoopEnd()
    seq_gen.TimerWaitAndRestart(integdelay)
    seq_gen.SetIntegratorMode(abs=integrate_with_fixed_range, ref=integrate_with_fixed_range, pmt2=integrate_with_fixed_range, pmt1=integrate_with_fixed_range)
    seq_gen.TimerWait()
    # read out analog signals
    seq_gen.GetPulseCounterResult(channel=0, relative=False, resetCounter=True, cumulative=True, dword=False, addrPos=0, resultPos=0)
    seq_gen.GetPulseCounterResult(channel=1, relative=False, resetCounter=True, cumulative=True, dword=False, addrPos=0, resultPos=5)
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=1)
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=2)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=6)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=7)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=10)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=11)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=14)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=15)
    seq_gen.TimerWaitAndRestart(offtime)
    seq_gen.LoopEnd()
    seq_gen.SetIntegratorMode(abs=full_reset, ref=full_reset, pmt2=full_reset, pmt1=full_reset)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    measurement_unit.resultAddresses[op_id] = range(0, 18)
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)

    resultsum = [0] * 18
    for i in range(iterations):
        try:
            await measurement_unit.ExecuteMeasurement(op_id)
            await asyncio.sleep(delay)
            results = await measurement_unit.ReadMeasurementValues(op_id)
            for index in range(0, len(results)):
                resultsum[index] = resultsum[index] + results[index]
        except BaseException as ex:
                print(f"pmttest3a() failed: {ex}")
        await asyncio.sleep(delay)
    return resultsum


async def pmttest3al(iterations, flashes=1, flash_mode=1, frequency=100, hv1 = 0.4, hv2 = 0.5, dl1 = 0.25, dl2 = 0.25, pre_cnt_window=10):
    '''
    ml 211027: pmttest measurement
	50µs window, with flash, low range only
    10ns timer tick
    1µs = 100 ticks
    1ms = 100000 ticks
    '''
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit

    delay = 0.1
    await pmt1setdl(dl1)
    await pmt2setdl(dl2)
    await pmt1sethv(hv1)
    await pmt2sethv(hv2)
    await pmt1hvenable()
    await pmt2hvenable()
    await hvgate1on()
    await hvgate2on()

    ## IntegratorModes
    full_reset = 0b010
    low_range_reset = 0b011
    integrate_autorange = 0b100
    integrate_with_fixed_range = 0b101
    integrate_in_low_range = 0b110
    integrate_in_high_range = 0b111

    lowresetdelay = 5000  # 50µs
    meastime = 5000       # 50µs
    delaytime = 5000      # 50µs
    integdelay = 500      # 5µs
    #pre_cnt_window = 100    # 1µs
    cnt_window = meastime // pre_cnt_window
    #window_corse, window_fine = divmod(cnt_window, 65536)

    power = 0.8
    # Limit Freqency for High Power Mode
    if frequency > 500 and flash_mode == 1:
        frequency = 500

    # Set the Flash_Pwr
    await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampPower, power, timeout=1)

    # Set ontime based on Power Mode 
    if flash_mode == 1:
        ontime = 120000
        await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampHighPowerEnable, 1, timeout=1)
    else:
        ontime = 60000
        await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampHighPowerEnable, 0, timeout=1)

    offtime = int(1 / frequency * 1e8 - ontime)

    op_id = 'pmttest3al'
    seq_gen = meas_seq_generator()
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=0)   # pmt1, count_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=1)   # pmt1, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=2)   # pmt1, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=3)   # pmt1, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=4)   # pmt1, analog_high_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=5)   # pmt2, count_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=6)   # pmt2, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=7)   # pmt2, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=8)   # pmt2, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=9)   # pmt2, analog_high_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=10)  # ref, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=11)  # ref, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=12)  # ref, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=13)  # ref, analog_high_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=14)  # abs, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=15)  # abs, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=16)  # abs, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=17)  # abs, analog_high_offset
    seq_gen.Loop(flashes)
    seq_gen.TimerWaitAndRestart(ontime)
    seq_gen.SetSignals(OutputSignal.Flash)
    seq_gen.SetIntegratorMode(abs=full_reset, ref=full_reset, pmt2=full_reset, pmt1=full_reset)
    seq_gen.TimerWaitAndRestart(lowresetdelay)
    seq_gen.SetIntegratorMode(abs=low_range_reset, ref=low_range_reset, pmt2=low_range_reset, pmt1=low_range_reset)
    seq_gen.TimerWaitAndRestart(delaytime)
    # high range offset
    seq_gen.TimerWaitAndRestart(meastime)
    seq_gen.SetIntegratorMode(abs=integrate_in_high_range, ref=integrate_in_high_range, pmt2=integrate_in_high_range, pmt1=integrate_in_high_range)
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)
    # channels: pmt1= 0; pmt2= 1; ref= 4; abs= 5
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=4)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=9)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=13)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=17)
    # low range offset
    seq_gen.TimerWaitAndRestart(meastime)
    seq_gen.SetIntegratorMode(abs=integrate_in_low_range, ref=integrate_in_low_range, pmt2=integrate_in_low_range, pmt1=integrate_in_low_range)
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=3)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=8)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=12)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=16)
    seq_gen.ResetSignals(OutputSignal.Flash)
    # measurement
    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    seq_gen.SetIntegratorMode(abs=integrate_in_low_range, ref=integrate_in_low_range, pmt2=integrate_in_low_range, pmt1=integrate_in_low_range)
    seq_gen.PulseCounterControl(channel=0, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    seq_gen.PulseCounterControl(channel=1, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    seq_gen.Loop(cnt_window)
    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    seq_gen.PulseCounterControl(channel=0, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
    seq_gen.PulseCounterControl(channel=1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
    seq_gen.LoopEnd()
    seq_gen.TimerWaitAndRestart(integdelay)
    seq_gen.SetIntegratorMode(abs=integrate_with_fixed_range, ref=integrate_with_fixed_range, pmt2=integrate_with_fixed_range, pmt1=integrate_with_fixed_range)
    seq_gen.TimerWait()
    # read out analog signals
    seq_gen.GetPulseCounterResult(channel=0, relative=False, resetCounter=True, cumulative=True, dword=False, addrPos=0, resultPos=0)
    seq_gen.GetPulseCounterResult(channel=1, relative=False, resetCounter=True, cumulative=True, dword=False, addrPos=0, resultPos=5)
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=1)
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=2)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=6)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=7)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=10)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=11)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=14)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=15)
    seq_gen.TimerWaitAndRestart(offtime)
    seq_gen.LoopEnd()
    seq_gen.SetIntegratorMode(abs=full_reset, ref=full_reset, pmt2=full_reset, pmt1=full_reset)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    measurement_unit.resultAddresses[op_id] = range(0, 18)
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)

    resultsum = [0] * 18
    for i in range(iterations):
        try:
            await measurement_unit.ExecuteMeasurement(op_id)
            await asyncio.sleep(delay)
            results = await measurement_unit.ReadMeasurementValues(op_id)
            for index in range(0, len(results)):
                resultsum[index] = resultsum[index] + results[index]
        except BaseException as ex:
                print(f"pmttest3al() failed: {ex}")
        await asyncio.sleep(delay)
    return resultsum

async def pmttest3ah(iterations, flashes=1, flash_mode=1, frequency=100, hv1 = 0.4, hv2 = 0.5, dl1 = 0.25, dl2 = 0.25, pre_cnt_window=10):
    '''
    ml 211027: pmttest measurement
	50µs window, with flash, high range only
    10ns timer tick
    1µs = 100 ticks
    1ms = 100000 ticks
    '''
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit

    delay = 0.1
    await pmt1setdl(dl1)
    await pmt2setdl(dl2)
    await pmt1sethv(hv1)
    await pmt2sethv(hv2)
    await pmt1hvenable()
    await pmt2hvenable()
    await hvgate1on()
    await hvgate2on()

    ## IntegratorModes
    full_reset = 0b010
    low_range_reset = 0b011
    integrate_autorange = 0b100
    integrate_with_fixed_range = 0b101
    integrate_in_low_range = 0b110
    integrate_in_high_range = 0b111

    lowresetdelay = 5000  # 50µs
    meastime = 5000       # 50µs
    delaytime = 5000      # 50µs
    integdelay = 500      # 5µs
    #pre_cnt_window = 100    # 1µs
    cnt_window = meastime // pre_cnt_window
    #window_corse, window_fine = divmod(cnt_window, 65536)

    power = 0.8
    # Limit Freqency for High Power Mode
    if frequency > 500 and flash_mode == 1:
        frequency = 500

    # Set the Flash_Pwr
    await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampPower, power, timeout=1)

    # Set ontime based on Power Mode 
    if flash_mode == 1:
        ontime = 120000
        await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampHighPowerEnable, 1, timeout=1)
    else:
        ontime = 60000
        await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampHighPowerEnable, 0, timeout=1)

    offtime = int(1 / frequency * 1e8 - ontime)

    op_id = 'pmttest3ah'
    seq_gen = meas_seq_generator()
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=0)   # pmt1, count_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=1)   # pmt1, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=2)   # pmt1, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=3)   # pmt1, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=4)   # pmt1, analog_high_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=5)   # pmt2, count_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=6)   # pmt2, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=7)   # pmt2, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=8)   # pmt2, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=9)   # pmt2, analog_high_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=10)  # ref, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=11)  # ref, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=12)  # ref, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=13)  # ref, analog_high_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=14)  # abs, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=15)  # abs, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=16)  # abs, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=17)  # abs, analog_high_offset
    seq_gen.Loop(flashes)
    seq_gen.TimerWaitAndRestart(ontime)
    seq_gen.SetSignals(OutputSignal.Flash)
    seq_gen.SetIntegratorMode(abs=full_reset, ref=full_reset, pmt2=full_reset, pmt1=full_reset)
    seq_gen.TimerWaitAndRestart(lowresetdelay)
    seq_gen.SetIntegratorMode(abs=low_range_reset, ref=low_range_reset, pmt2=low_range_reset, pmt1=low_range_reset)
    seq_gen.TimerWaitAndRestart(delaytime)
    # high range offset
    seq_gen.TimerWaitAndRestart(meastime)
    seq_gen.SetIntegratorMode(abs=integrate_in_high_range, ref=integrate_in_high_range, pmt2=integrate_in_high_range, pmt1=integrate_in_high_range)
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)
    # channels: pmt1= 0; pmt2= 1; ref= 4; abs= 5
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=4)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=9)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=13)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=17)
    # low range offset
    seq_gen.TimerWaitAndRestart(meastime)
    seq_gen.SetIntegratorMode(abs=integrate_in_low_range, ref=integrate_in_low_range, pmt2=integrate_in_low_range, pmt1=integrate_in_low_range)
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=3)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=8)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=12)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=16)
    seq_gen.ResetSignals(OutputSignal.Flash)
    # measurement
    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    seq_gen.SetIntegratorMode(abs=integrate_in_high_range, ref=integrate_in_high_range, pmt2=integrate_in_high_range, pmt1=integrate_in_high_range)
    seq_gen.PulseCounterControl(channel=0, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    seq_gen.PulseCounterControl(channel=1, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    seq_gen.Loop(cnt_window)
    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    seq_gen.PulseCounterControl(channel=0, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
    seq_gen.PulseCounterControl(channel=1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
    seq_gen.LoopEnd()
    seq_gen.TimerWaitAndRestart(integdelay)
    seq_gen.SetIntegratorMode(abs=integrate_with_fixed_range, ref=integrate_with_fixed_range, pmt2=integrate_with_fixed_range, pmt1=integrate_with_fixed_range)
    seq_gen.TimerWait()
    # read out analog signals
    seq_gen.GetPulseCounterResult(channel=0, relative=False, resetCounter=True, cumulative=True, dword=False, addrPos=0, resultPos=0)
    seq_gen.GetPulseCounterResult(channel=1, relative=False, resetCounter=True, cumulative=True, dword=False, addrPos=0, resultPos=5)
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=1)
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=2)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=6)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=7)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=10)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=11)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=14)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=15)
    seq_gen.TimerWaitAndRestart(offtime)
    seq_gen.LoopEnd()
    seq_gen.SetIntegratorMode(abs=full_reset, ref=full_reset, pmt2=full_reset, pmt1=full_reset)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    measurement_unit.resultAddresses[op_id] = range(0, 18)
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)

    resultsum = [0] * 18
    for i in range(iterations):
        try:
            await measurement_unit.ExecuteMeasurement(op_id)
            await asyncio.sleep(delay)
            results = await measurement_unit.ReadMeasurementValues(op_id)
            for index in range(0, len(results)):
                resultsum[index] = resultsum[index] + results[index]
        except BaseException as ex:
                print(f"pmttest3ah() failed: {ex}")
        await asyncio.sleep(delay)
    return resultsum

async def pmttest3b(iterations, flashes=1, flash_mode=1, frequency=100, hv1 = 0.4, hv2 = 0.5, dl1 = 0.25, dl2 = 0.25, pre_cnt_window=10):
    '''
    ml 211027: pmttest measurement
	50µs Window, without Flash
    10ns timer tick
    1µs = 100 ticks
    1ms = 100000 ticks
    '''
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit

    delay = 0.1
    await pmt1setdl(dl1)
    await pmt2setdl(dl2)
    await pmt1sethv(hv1)
    await pmt2sethv(hv2)
    await pmt1hvenable()
    await pmt2hvenable()
    await hvgate1on()
    await hvgate2on()

    ## IntegratorModes
    full_reset = 0b010
    low_range_reset = 0b011
    integrate_autorange = 0b100
    integrate_with_fixed_range = 0b101
    integrate_in_low_range = 0b110
    integrate_in_high_range = 0b111

    lowresetdelay = 5000  # 50µs
    meastime = 5000       # 50µs
    delaytime = 5000      # 50µs
    integdelay = 500      # 5µs
    #pre_cnt_window = 100    # 1µs
    cnt_window = meastime // pre_cnt_window
    #window_corse, window_fine = divmod(cnt_window, 65536)

    power = 0.8
    # Limit Freqency for High Power Mode
    if frequency > 500 and flash_mode == 1:
        frequency = 500

    # Set the Flash_Pwr
    await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampPower, power, timeout=1)

    # Set ontime based on Power Mode 
    if flash_mode == 1:
        ontime = 120000
        await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampHighPowerEnable, 0, timeout=1)
    else:
        ontime = 60000
        await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampHighPowerEnable, 1, timeout=1)

    offtime = int(1 / frequency * 1e8 - ontime)

    op_id = 'pmttest3b'
    seq_gen = meas_seq_generator()
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=0)   # pmt1, count_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=1)   # pmt1, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=2)   # pmt1, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=3)   # pmt1, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=4)   # pmt1, analog_high_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=5)   # pmt2, count_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=6)   # pmt2, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=7)   # pmt2, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=8)   # pmt2, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=9)   # pmt2, analog_high_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=10)  # ref, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=11)  # ref, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=12)  # ref, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=13)  # ref, analog_high_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=14)  # abs, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=15)  # abs, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=16)  # abs, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=17)  # abs, analog_high_offset
    seq_gen.Loop(flashes)
    seq_gen.TimerWaitAndRestart(ontime)
    #seq_gen.SetSignals(OutputSignal.Flash)
    seq_gen.SetIntegratorMode(abs=full_reset, ref=full_reset, pmt2=full_reset, pmt1=full_reset)
    seq_gen.TimerWaitAndRestart(lowresetdelay)
    seq_gen.SetIntegratorMode(abs=low_range_reset, ref=low_range_reset, pmt2=low_range_reset, pmt1=low_range_reset)
    seq_gen.TimerWaitAndRestart(delaytime)
    # high range offset
    seq_gen.TimerWaitAndRestart(meastime)
    seq_gen.SetIntegratorMode(abs=integrate_in_high_range, ref=integrate_in_high_range, pmt2=integrate_in_high_range, pmt1=integrate_in_high_range)
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)
    # channels: pmt1= 0; pmt2= 1; ref= 4; abs= 5
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=4)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=9)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=13)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=17)
    # low range offset
    seq_gen.TimerWaitAndRestart(meastime)
    seq_gen.SetIntegratorMode(abs=integrate_in_low_range, ref=integrate_in_low_range, pmt2=integrate_in_low_range, pmt1=integrate_in_low_range)
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=3)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=8)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=12)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=16)
    #seq_gen.ResetSignals(OutputSignal.Flash)
    # measurement
    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    seq_gen.SetIntegratorMode(abs=integrate_autorange, ref=integrate_autorange, pmt2=integrate_autorange, pmt1=integrate_autorange)
    seq_gen.PulseCounterControl(channel=0, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    seq_gen.PulseCounterControl(channel=1, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    seq_gen.Loop(cnt_window)
    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    seq_gen.PulseCounterControl(channel=0, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
    seq_gen.PulseCounterControl(channel=1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
    seq_gen.LoopEnd()
    seq_gen.TimerWaitAndRestart(integdelay)
    seq_gen.SetIntegratorMode(abs=integrate_with_fixed_range, ref=integrate_with_fixed_range, pmt2=integrate_with_fixed_range, pmt1=integrate_with_fixed_range)
    seq_gen.TimerWait()
    # read out analog signals
    seq_gen.GetPulseCounterResult(channel=0, relative=False, resetCounter=True, cumulative=True, dword=False, addrPos=0, resultPos=0)
    seq_gen.GetPulseCounterResult(channel=1, relative=False, resetCounter=True, cumulative=True, dword=False, addrPos=0, resultPos=5)
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=1)
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=2)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=6)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=7)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=10)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=11)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=14)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=15)
    seq_gen.TimerWaitAndRestart(offtime)
    seq_gen.LoopEnd()
    seq_gen.SetIntegratorMode(abs=full_reset, ref=full_reset, pmt2=full_reset, pmt1=full_reset)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    measurement_unit.resultAddresses[op_id] = range(0, 18)
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)

    resultsum = [0] * 18
    for i in range(iterations):
        try:
            await measurement_unit.ExecuteMeasurement(op_id)
            await asyncio.sleep(delay)
            results = await measurement_unit.ReadMeasurementValues(op_id)
            for index in range(0, len(results)):
                resultsum[index] = resultsum[index] + results[index]
        except BaseException as ex:
                print(f"pmttest3b() failed: {ex}")
        await asyncio.sleep(delay)
    return resultsum


async def as1scana(start=500, stop=-4200, step=10, iterations=10, flashes=1, flash_mode=1, frequency=100, hv1 = 0.4, hv2 = 0.5, dl1 = 0.25, dl2 = 0.25, pre_cnt_window=10 ):
    '''
    ml 211027:
    '''
    meastime = 5000       # 50µs
    cnt_window = meastime // pre_cnt_window

    await as1_home()
    dfp = datapath('AS1Scan')
    path, filename = os.path.split(dfp)
    result = str.format("Filename: {0}; Start: {1}; Stop: {2}; Step: {3}; Iterations: {4}; Flashes: {5}; Flash-Mode: {6}; Frequency: {7}",filename, start, stop, step, iterations, flashes, flash_mode, frequency)
    report(result, '.alpha.as1scana')
    result = str.format("HV1: {0}; HV2: {1}; DL1: {2}; DL2: {3}; CntWindow: {4};PreCntWindow: {5}", hv1, hv2, dl1, dl2, cnt_window, pre_cnt_window)
    report(result, '.alpha.as1scana')
    print (filename)
    with open(dfp, 'a') as data:
        data.write(f"pos\tpmt1_cr\tpmt1_alr\tpmt1_ahr\tpmt1_alo\tpmt1_aho\tpmt2_cr\tpmt2_alr\tpmt2_ahr\tpmt2_alo\tpmt2_aho\tref_alr\tref_ahr\tref_alo\tref_aho\tabs_alr\tabs_ahr\tabs_alo\tabs_aho\n")
        for i in range(start, stop, step):
            pos = i*0.001
            await as1(pos)
            await as2(pos)
            results = await pmttest3a(iterations, flashes, flash_mode, frequency, hv1, hv2, dl1, dl2, pre_cnt_window)
            data.write(f"{pos}\t{results[0]}\t{results[1]}\t{results[2]}\t{results[3]}\t{results[4]}\t{results[5]}\t{results[6]}\t{results[7]}\t{results[8]}\t{results[9]}\t{results[10]}\t{results[11]}\t{results[12]}\t{results[13]}\t{results[14]}\t{results[15]}\t{results[16]}\t{results[17]}\n")
    await as1_home()
    await as2_home()
    print('AS1_Scan Done\n')

async def as1scanal(start=500, stop=-4200, step=10, iterations=10, flashes=1, flash_mode=1, frequency=100, hv1 = 0.4, hv2 = 0.5, dl1 = 0.25, dl2 = 0.25, pre_cnt_window=10 ):
    '''
    ml 211027:
    '''
    meastime = 5000       # 50µs
    cnt_window = meastime // pre_cnt_window

    await as1_home()
    dfp = datapath('AS1Scan')
    path, filename = os.path.split(dfp)
    result = str.format("Filename: {0}; Start: {1}; Stop: {2}; Step: {3}; Iterations: {4}; Flashes: {5}; Flash-Mode: {6}; Frequency: {7}",filename, start, stop, step, iterations, flashes, flash_mode, frequency)
    report(result, '.alpha.as1scanal')
    result = str.format("HV1: {0}; HV2: {1}; DL1: {2}; DL2: {3}; CntWindow: {4};PreCntWindow: {5}", hv1, hv2, dl1, dl2, cnt_window, pre_cnt_window)
    report(result, '.alpha.as1scanal')
    print (filename)
    with open(dfp, 'a') as data:
        data.write(f"pos\tpmt1_cr\tpmt1_alr\tpmt1_ahr\tpmt1_alo\tpmt1_aho\tpmt2_cr\tpmt2_alr\tpmt2_ahr\tpmt2_alo\tpmt2_aho\tref_alr\tref_ahr\tref_alo\tref_aho\tabs_alr\tabs_ahr\tabs_alo\tabs_aho\n")
        for i in range(start, stop, step):
            pos = i*0.001
            await as1(pos)
            await as2(pos)
            results = await pmttest3al(iterations, flashes, flash_mode, frequency, hv1, hv2, dl1, dl2, pre_cnt_window)
            data.write(f"{pos}\t{results[0]}\t{results[1]}\t{results[2]}\t{results[3]}\t{results[4]}\t{results[5]}\t{results[6]}\t{results[7]}\t{results[8]}\t{results[9]}\t{results[10]}\t{results[11]}\t{results[12]}\t{results[13]}\t{results[14]}\t{results[15]}\t{results[16]}\t{results[17]}\n")
    await as1_home()
    await as2_home()
    print('AS1_ScanL Done\n')

async def as1scanah(start=500, stop=-4200, step=10, iterations=10, flashes=1, flash_mode=1, frequency=100, hv1 = 0.4, hv2 = 0.5, dl1 = 0.25, dl2 = 0.25, pre_cnt_window=10 ):
    '''
    ml 211027:
    '''
    meastime = 5000       # 50µs
    cnt_window = meastime // pre_cnt_window

    await as1_home()
    dfp = datapath('AS1Scan')
    path, filename = os.path.split(dfp)
    result = str.format("Filename: {0}; Start: {1}; Stop: {2}; Step: {3}; Iterations: {4}; Flashes: {5}; Flash-Mode: {6}; Frequency: {7}",filename, start, stop, step, iterations, flashes, flash_mode, frequency)
    report(result, '.alpha.as1scanah')
    result = str.format("HV1: {0}; HV2: {1}; DL1: {2}; DL2: {3}; CntWindow: {4};PreCntWindow: {5}", hv1, hv2, dl1, dl2, cnt_window, pre_cnt_window)
    report(result, '.alpha.as1scanah')
    print (filename)
    with open(dfp, 'a') as data:
        data.write(f"pos\tpmt1_cr\tpmt1_alr\tpmt1_ahr\tpmt1_alo\tpmt1_aho\tpmt2_cr\tpmt2_alr\tpmt2_ahr\tpmt2_alo\tpmt2_aho\tref_alr\tref_ahr\tref_alo\tref_aho\tabs_alr\tabs_ahr\tabs_alo\tabs_aho\n")
        for i in range(start, stop, step):
            pos = i*0.001
            await as1(pos)
            await as2(pos)
            results = await pmttest3ah(iterations, flashes, flash_mode, frequency, hv1, hv2, dl1, dl2, pre_cnt_window)
            data.write(f"{pos}\t{results[0]}\t{results[1]}\t{results[2]}\t{results[3]}\t{results[4]}\t{results[5]}\t{results[6]}\t{results[7]}\t{results[8]}\t{results[9]}\t{results[10]}\t{results[11]}\t{results[12]}\t{results[13]}\t{results[14]}\t{results[15]}\t{results[16]}\t{results[17]}\n")
    await as1_home()
    await as2_home()
    print('AS1_ScanH Done\n')
        
async def as1scanb(start=500, stop=-4200, step=10, iterations=10, flashes=1, flash_mode=1, frequency=100, hv1 = 0.4, hv2 = 0.5, dl1 = 0.25, dl2 = 0.25, pre_cnt_window=10 ):
    '''
    ml 211027:
    '''
    meastime = 5000       # 50µs
    cnt_window = meastime // pre_cnt_window

    await as1_home()
    dfp = datapath('AS1Scan')
    path, filename = os.path.split(dfp)
    result = str.format("Filename: {0}; Start: {1}; Stop: {2}; Step: {3}; Iterations: {4}; Flashes: {5}; Flash-Mode: {6}; Frequency: {7}",filename, start, stop, step, iterations, flashes, flash_mode, frequency)
    report(result, '.alpha.as1scanb')
    result = str.format("HV1: {0}; HV2: {1}; DL1: {2}; DL2: {3}; CntWindow: {4};PreCntWindow: {5}", hv1, hv2, dl1, dl2, cnt_window, pre_cnt_window)
    report(result, '.alpha.as1scanb')
    print (filename)
    with open(dfp, 'a') as data:
        data.write(f"pos\tpmt1_cr\tpmt1_alr\tpmt1_ahr\tpmt1_alo\tpmt1_aho\tpmt2_cr\tpmt2_alr\tpmt2_ahr\tpmt2_alo\tpmt2_aho\tref_alr\tref_ahr\tref_alo\tref_aho\tabs_alr\tabs_ahr\tabs_alo\tabs_aho\n")
        for i in range(start, stop, step):
            pos = i*0.001
            await as1(pos)
            results = await pmttest3b(iterations, flashes, flash_mode, frequency, hv1, hv2, dl1, dl2, pre_cnt_window)
            data.write(f"{pos}\t{results[0]}\t{results[1]}\t{results[2]}\t{results[3]}\t{results[4]}\t{results[5]}\t{results[6]}\t{results[7]}\t{results[8]}\t{results[9]}\t{results[10]}\t{results[11]}\t{results[12]}\t{results[13]}\t{results[14]}\t{results[15]}\t{results[16]}\t{results[17]}\n")
    await as1_home()
    print('AS1_Scan Done\n')
        


async def hartasetup():
    await st_measpos()
    await st_platetype("96_Opti_Plate")
    await fms_set(1)
    report('fms_set(1)', '.alpha.hartasetup')
    await fm(4)
    report('fm(4)', '.alpha.hartasetup')
    await as1_3()
    report('as1_3', '.alpha.hartasetup')
    xw=0
    yw=0
    await st_movetowell(xw, yw)
    result = str.format("Well A{0}", yw+1) 
    report(result, '.alpha.hartasetup')
    await as1_close()
    report('as1_close', '.alpha.hartasetup')

async def hartatest(dlstart=1200, dlstop=3000, dlstep=200, hvstart=1500, hvstop=3000, hvstep=1, window=100, awindow=1000, z=4, xo=8, yo=-8):
    delay=0.1  
    await st_measpos()
    await st_platetype("96_Opti_Plate")
    await fms_set(1)
    report('fms_set(1)', '.alpha.hartatest')
    await fm(z)
    await as1_3()
    report('as1_3', '.alpha.hartatest')
    result = str.format("X-Offset: {0}, Y-Offset: {1}, FM-Z: {2}", xo, yo, z) 
    report(result, '.alpha.hartatest')
    yw=0
    for xw in range(12):
        await st_movetowell(xw, yw, xo, yo)
        result = str.format("Well A{0}", xw+1) 
        report(result, '.alpha.hartatest')
        await dlhvscan1c(dlstart, dlstop, dlstep, hvstart, hvstop, hvstep, window, awindow)
        await asyncio.sleep(delay)
    await as1_close()
    report('as1_close', '.alpha.hartatest')

async def nivotest(dl=1200, hv=2500, window=100, awstart=10, awstop=1000, awstep=10, z=4, xo=8, yo=-8):
    delay=0.1  
    await st_measpos()
    await st_platetype("96_Opti_Plate")
    await fms_set(1)
    report('fms_set(1)', '.alpha.nivotest')
    await fm(z)
    await as1_3()
    report('as1_3', '.alpha.nivotest')
    result = str.format("X-Offset: {0}, Y-Offset: {1}, FM-Z: {2}", xo, yo, z) 
    report(result, '.alpha.nivotest')
    await st_movetowell(2, 0, xo, yo)
    report('Well A3', '.alpha.nivotest')
    await awinscan1c(dl, hv, window, awstart, awstop, awstep)
    await asyncio.sleep(delay)
    await as1_close()
    report('as1_close', '.alpha.nivotest')

async def nivoinit(z=4):
    await st_measpos()
    await st_platetype("96_Nivo_Test_Plate")
    await els_2()
    await fms_set(2)
    report('fms_set(2)', '.alpha.nivofitest')
    await fm(z)
    await as1_3()
    report('as1_3', '.alpha.nivofitest')
    await hvpmt1on()
    await hvgate1(True)

async def nivoexit():
    await as1_close()
    report('as1_close', '.alpha.nivofitest')
    await hvgate1(False)
    await hvpmt1off()

# Alpha Functions

async def al_on():
    address = 0
    sequence = [
            0x03000000 | signals['alpha'],    # Reset Signal to turn Laser On!!!!
            0x00000000,
        ]
    await write_sequence(address, sequence)
    await start_sequence(address)
    print('=============Alpha-Laser On!!!==================')

async def al_off():
    address = 0
    sequence = [
        0x02000000 | signals['alpha'],    # Set Signal to turn Laser Off!!!
        0x00000000,
        ]
    await write_sequence(address, sequence)
    await start_sequence(address)
    print('=============Alpha-Laser Off!!!==================')

async def al_init():
    await al_off()
    #await fms_set(4)
    await els_a()
    await st_platetype("384_Opti_Plate")
    await st_measpos()
    await hw.al_enable()
    
async def ws_init():
    await al_off()
    await fms_diag()
    await els_2()
    await bld_pd()
    await st_measpos()
    
async def go():
    global rfp
    rfp = reportpath()
    report('Init', 'alpha_go')
    await init()
    await hw.start_al()
    await al_init()

async def go2():
    global rfp
    rfp = reportpath()
    report('Init', 'alpha_go2')
    await init()
    await hw.start_al()
    await ws_init()


async def aldec_2(x = 1, y = 1, alphaontime = 100, alphadelay = 10, alphaofftime = 100, windows = 5, pmt1hv=1800, discr1=2000, xo =0, yo =0):
    ontime = int(alphaontime * 1e5)         # TimerTick 10ns
    delaytime = int(alphadelay * 1e5)       # TimerTick 10ns
    cnt_window = int(alphaofftime)       
    pre_cnt_window = 100000                 # 1e5 TimerTicks a 10ns = 1ms
    address = 0
    sequence = [
        0x02000000 | signals['alpha'],      # SetSignal Alpha / Off
        0x7C000000 | (ontime-1),            # Start Timer for XXms
        0x03000000 | signals['alpha'],      # ResetSignsl Alpha / On
        0x7C000000 | (delaytime-1),         # Start Timer for XXms
        0x02000000 | signals['alpha'],      # SetSignal Alpha / Off
        0x08000000,                         # SetAddrReg(DataNotAddrSrc=0, rel=0, sign=0, StackNotRegSrc=0, srcReg=0, dstReg=0, addr=0)
        0x07000000 | (windows),             # Loop(windows)                         
        0x7C000000 | (pre_cnt_window - 1),  #     TimerWaitAndRestart(pre_cnt_window - 1)
        0x88B00000,                         #     PulseCounterControl(ch=0, add=0, RstCnt=1, RstPreCnt=1, corr=0)                           
        0x07000000 | (cnt_window - 1),      #     Loop(cnt_window - 1)
        0x7C000000 | (pre_cnt_window - 1),  #         TimerWaitAndRestart(pre_cnt_window - 1)
        0x88D00000,                         #         PulseCounterControl(ch=0, add=1, RstCnt=0, RstPreCnt=1, corr=0)
        0x05000000,                         #     LoopEnd(cnt_windows)
        0x7C000000 | (100),                 #     TimerWaitAndRestart(100)
        0x88D00000,                         #     PulseCounterControl(ch=0, add=1, RstCnt=0, RstPreCnt=1, corr=0)
        0x80CC0000,                         #     GetPulseCounterResult(ch=0, rel=1, RstCnt=0, add=1, dword=1, addrReg=0, addr=0)
        0x08400001,                         #     SetAddrReg(DataNotAddrSrc=0, rel=1, sign=0, StackNotRegSrc=0, srcReg=0, dstReg=0, addr=1)
        0x05000000,                         # LoopEnd(windows)
        0x7C000000 | (100),                 # TimerWaitAndRestart(100)
        0x00000000,                         # Stop(0)
    ]
    await hw.pmt1_setdl( discr1)
    await hw.pmt1_sethv( pmt1hv)
    await hvpmt1on()
    await hvgate1(enable=True)
    await st_movetowell( x-1, y-1, xo, yo)
    await write_sequence(address, sequence)
    sleepdelay = 0.5+windows*alphaofftime*0.001
    result = str.format("Column: {0}, Row: {1}, AlphaOnTime: {2}, AlphaDelay: {3}, AlphaOffTime: {4}, Windows: {5}, PMT1-HV: {6}, Discr1: {7}, X-Offset: {8}, Y-Offset: {9}",x, y, alphaontime, alphadelay, alphaofftime, windows, pmt1hv, discr1, xo, yo)
    report(result, '.alpha.aldec_2')
    dfp = datapath('AlphaTest')
    path, filename = os.path.split(dfp)
    report(filename, '.alpha.aldec_2'); #Write to Report
    print (filename)
    with open(dfp, 'a') as data:
        data.write("Time_Stamp  \tAlphaOnTime \tAlphaDelay \tAlphaOffTime \tWindows \tX \tY \tX-Pos \tY-Pos \tCounts\n")
        stpos = await st_getposition()
        xpos = stpos[0]
        ypos = stpos[1]
        time_stamp = dt.now().strftime('%Y%m%d %H:%M:%S')
        data.write(f"{time_stamp}  \t{alphaontime} \t{alphadelay} \t{alphaofftime} \t{windows} \t{x} \t{y} \t{xpos} \t{ypos}")
        try:
            await start_sequence(address)
            await asyncio.sleep(sleepdelay)
            results = await get_results(address=0, size=windows)
            await asyncio.sleep(sleepdelay)
            print(f"{results}")
            data.write(f"\t{results}\n")
        except BaseException as ex:
            print(f"alphacount() failed: {ex}")
            data.write(f"alphacount() failed: {ex}\n")
    await hvgate1(enable=False)
    await hvpmt1off()
    print('=============AlphaDecayTest2==================')

async def aldec_2b(x = 1, y = 1, alphaontime = 100, alphadelay = 10, alphaofftime = 100, windows = 5, pmt1hv=1800, discr1=2000, xo =0, yo =0):
    ontime = int(alphaontime * 1e5)         # TimerTick 10ns
    delaytime = int(alphadelay * 1e5)       # TimerTick 10ns
    cnt_window = int(alphaofftime)       
    pre_cnt_window = 100000                 # 1e5 TimerTicks a 10ns = 1ms

    op_id = 'alphadecay'
    seq_gen = meas_seq_generator()
    seq_gen.SetSignals(Alpha)    # off
    seq_gen.TimerWaitAndRestart(ontime)
    seq_gen.ResetSignals(Alpha)  # on
    seq_gen.TimerWaitAndRestart(delaytime)
    seq_gen.SetSignals(Alpha)    # off
    seq_gen.SetAddrReg(relative=0, dataNotAddrSrc=0, sign=0, stackNotRegSrc=0, srcReg=0, dstReg=0, addr=0)
    seq_gen.Loop(windows)
    seq_gen.TimerWaitAndRestart(pre_cnt_windows)
    seq_gen.PulseCounterControl(channel=0, cumulative=0, resetCounter=1, resetPresetCounter=1, correctionOn=0)
    seq_gen.Loop(cnt_windows)
    seq_gen.TimerWaitAndRestart(pre_cnt_windows)
    seq_gen.PulseCounterControl(channel=0, cumulative=1, resetCounter=0, resetPresetCounter=1, correctionOn=0)
    seq_gen.LoopEnd	# cnt_windows
    seq_gen.TimerWaitAndRestart(100)
    seq_gen.PulseCounterControl(channel=0, cumulative=1, resetCounter=0, resetPresetCounter=1, correctionOn=0)
    seq_gen.GetPulseCounterResult(channel=0, relative=1, resetCounter=0, cumulative=1, dword=1, addrPos=0, resultPos=0)
    seq_gen.SetAddrReg(relative=1, dataNotAddrSrc=0, sign=0, stackNotRegSrc=0, srcReg=0, dstReg=0, addr=1)
    seq_gen.LoopEnd	# windows
    seq_gen.TimerWaitAndRestart(100)
    seq_gen.Stop(0)

    await hw.pmt1_setdl( discr1)
    await hw.pmt1_sethv( pmt1hv)
    await hvpmt1on()
    await hvgate1(enable=True)
    await st_movetowell( x-1, y-1, xo, yo)

    measurement_unit.ClearOperations()
    measurement_unit.resultAddresses[op_id] = range(0, windows)
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)

    result = str.format("Column: {0}, Row: {1}, AlphaOnTime: {2}, AlphaDelay: {3}, AlphaOffTime: {4}, Windows: {5}, PMT1-HV: {6}, Discr1: {7}, X-Offset: {8}, Y-Offset: {9}",x, y, alphaontime, alphadelay, alphaofftime, windows, pmt1hv, discr1, xo, yo)
    report(result, '.alpha.aldec_2')
    dfp = datapath('AlphaTest')
    path, filename = os.path.split(dfp)
    report(filename, '.alpha.aldec_2'); #Write to Report
    print (filename)
    with open(dfp, 'a') as data:
        data.write("Time_Stamp  \tAlphaOnTime \tAlphaDelay \tAlphaOffTime \tWindows \tX \tY \tX-Pos \tY-Pos \tCounts\n")
        stpos = await st_getposition()
        xpos = stpos[0]
        ypos = stpos[1]
        time_stamp = dt.now().strftime('%Y%m%d %H:%M:%S')
        data.write(f"{time_stamp}  \t{alphaontime} \t{alphadelay} \t{alphaofftime} \t{windows} \t{x} \t{y} \t{xpos} \t{ypos}")
        try:
            await measurement_unit.ExecuteMeasurement(op_id)
            results = await measurement_unit.ReadMeasurementValues(op_id)
            await asyncio.sleep(sleepdelay)
            print(f"{results}")
            data.write(f"\t{results}\n")
        except BaseException as ex:
            print(f"alphacount() failed: {ex}")
            data.write(f"alphacount() failed: {ex}\n")
    await hvgate1(enable=False)
    await hvpmt1off()
    print('=============AlphaDecayTest2==================')


# Reference Photodiode Functions

async def bldscana(start=0, stop=3600, step=150, iterations=10, flashes=1, flash_mode=1, frequency=100):
    '''
    ml 201001:
    rework 210806    
    '''
    await bld_home()
    dfp = datapath('BLDScan')
    path, filename = os.path.split(dfp)
    result = str.format("Filename: {0}; Start: {1}; Stop: {2}; Step: {3}; Iterations: {4}; Flashes: {5}; Flash-Mode: {6}; Frequency: {7}",filename, start, stop, step, iterations, flashes, flash_mode, frequency)
    report(result, '.alpha.bldscana')
    print (filename)
    with open(dfp, 'a') as data:
        data.write(f"pos\tBGRefLo\tBGRefHi\tBGAbsLo\tBGAbsHi\tRefLo\tRefHi\tAbsLo\tAbsHi\n")
        for i in range(start, stop, step):
            pos = i*0.1
            await bld(pos)
            results = await reftest2a(iterations, flashes, flash_mode, frequency)
            data.write(f"{pos}\t{results[0]}\t{results[1]}\t{results[2]}\t{results[3]}\t{results[4]}\t{results[5]}\t{results[6]}\t{results[7]}\n")
    await bld_home()
    print('BLD_ScanA Done\n')
        
async def bldscanal(start=0, stop=3600, step=150, iterations=10, flashes=1, flash_mode=1, frequency=100):
    '''
    ml 201001:
    rework 210806    
    '''
    await bld_home()
    dfp = datapath('BLDScan')
    path, filename = os.path.split(dfp)
    result = str.format("Filename: {0}; Start: {1}; Stop: {2}; Step: {3}; Iterations: {4}; Flashes: {5}; Flash-Mode: {6}; Frequency: {7}",filename, start, stop, step, iterations, flashes, flash_mode, frequency)
    report(result, '.alpha.bldscanal')
    print (filename)
    with open(dfp, 'a') as data:
        data.write(f"pos\tBGRefLo\tBGRefHi\tBGAbsLo\tBGAbsHi\tRefLo\tRefHi\tAbsLo\tAbsHi\n")
        for i in range(start, stop, step):
            pos = i*0.1
            await bld(pos)
            results = await reftest2al(iterations, flashes, flash_mode, frequency)
            data.write(f"{pos}\t{results[0]}\t{results[1]}\t{results[2]}\t{results[3]}\t{results[4]}\t{results[5]}\t{results[6]}\t{results[7]}\n")
    await bld_home()
    print('BLD_ScanAl Done\n')
        
async def bldscanah(start=0, stop=3600, step=150, iterations=10, flashes=1, flash_mode=1, frequency=100):
    '''
    ml 201001:
    rework 210806    
    '''
    await bld_home()
    dfp = datapath('BLDScan')
    path, filename = os.path.split(dfp)
    result = str.format("Filename: {0}; Start: {1}; Stop: {2}; Step: {3}; Iterations: {4}; Flashes: {5}; Flash-Mode: {6}; Frequency: {7}",filename, start, stop, step, iterations, flashes, flash_mode, frequency)
    report(result, '.alpha.bldscanah')
    print (filename)
    with open(dfp, 'a') as data:
        data.write(f"pos\tBGRefLo\tBGRefHi\tBGAbsLo\tBGAbsHi\tRefLo\tRefHi\tAbsLo\tAbsHi\n")
        for i in range(start, stop, step):
            pos = i*0.1
            await bld(pos)
            results = await reftest2ah(iterations, flashes, flash_mode, frequency)
            data.write(f"{pos}\t{results[0]}\t{results[1]}\t{results[2]}\t{results[3]}\t{results[4]}\t{results[5]}\t{results[6]}\t{results[7]}\n")
    await bld_home()
    print('BLD_ScanAh Done\n')
        
async def bldscanb(start=0, stop=3600, step=150, iterations=10, flashes=1, flash_mode=1, frequency=100):
    '''
    ml 201001:
    rework 210806    
    '''
    await bld_home()
    dfp = datapath('BLDScan')
    path, filename = os.path.split(dfp)
    result = str.format("Filename: {0}; Start: {1}; Stop: {2}; Step: {3}; Iterations: {4}; Flashes: {5}; Flash-Mode: {6}; Frequency: {7}",filename, start, stop, step, iterations, flashes, flash_mode, frequency)
    report(result, '.alpha.bldscanb')
    print (filename)
    with open(dfp, 'a') as data:
        data.write(f"pos\tBGRefLo\tBGRefHi\tBGAbsLo\tBGAbsHi\tRefLo\tRefHi\tAbsLo\tAbsHi\n")
        for i in range(start, stop, step):
            pos = i*0.1
            await bld(pos)
            results = await reftest2b(iterations, flashes, flash_mode, frequency)
            data.write(f"{pos}\t{results[0]}\t{results[1]}\t{results[2]}\t{results[3]}\t{results[4]}\t{results[5]}\t{results[6]}\t{results[7]}\n")
    await bld_home()
    print('BLD_ScanB Done\n')
        
async def elsscana(start=0, stop=3600, step=150, iterations=10, flashes=1, flash_mode=1, frequency=100, power=0.5, scale=0.1):
    '''
    ml 201001:
    rework 210806    
    '''
    await els_home()
    dfp = datapath('ELSScan')
    path, filename = os.path.split(dfp)
    result = str.format("Filename: {0}; Start: {1}; Stop: {2}; Step: {3}; Iterations: {4}; Flashes: {5}; Flash-Mode: {6}; Frequency: {7}; Power: {8}",filename, start, stop, step, iterations, flashes, flash_mode, frequency, power)
    report(result, '.alpha.elsscana')
    print (filename)
    with open(dfp, 'a') as data:
        data.write(f"pos\tBGRefLo\tBGRefHi\tBGAbsLo\tBGAbsHi\tRefLo\tRefHi\tAbsLo\tAbsHi\n")
        for i in range(start, stop, step):
            pos = i*scale
            await els(pos)
            results = await reftest2a(iterations, flashes, flash_mode, frequency, power)
            data.write(f"{pos}\t{results[0]}\t{results[1]}\t{results[2]}\t{results[3]}\t{results[4]}\t{results[5]}\t{results[6]}\t{results[7]}\n")
    await els_home()
    print('ELS_Scan Done\n')
        
async def elsscanal(start=0, stop=3600, step=150, iterations=10, flashes=1, flash_mode=1, frequency=100, power=0.5, scale=0.1):
    '''
    ml 201001:
    rework 210806    
    '''
    await els_home()
    dfp = datapath('ELSScan')
    path, filename = os.path.split(dfp)
    result = str.format("Filename: {0}; Start: {1}; Stop: {2}; Step: {3}; Iterations: {4}; Flashes: {5}; Flash-Mode: {6}; Frequency: {7}; Power: {8}",filename, start, stop, step, iterations, flashes, flash_mode, frequency, power)
    report(result, '.alpha.elsscanal')
    print (filename)
    with open(dfp, 'a') as data:
        data.write(f"pos\tBGRefLo\tBGRefHi\tBGAbsLo\tBGAbsHi\tRefLo\tRefHi\tAbsLo\tAbsHi\n")
        for i in range(start, stop, step):
            pos = i*scale
            await els(pos)
            results = await reftest2al(iterations, flashes, flash_mode, frequency)
            data.write(f"{pos}\t{results[0]}\t{results[1]}\t{results[2]}\t{results[3]}\t{results[4]}\t{results[5]}\t{results[6]}\t{results[7]}\n")
    await els_home()
    print('ELS_Scan Done\n')
        
async def elsscanah(start=0, stop=3600, step=150, iterations=10, flashes=1, flash_mode=1, frequency=100, power=0.5, scale=0.1):
    '''
    ml 201001:
    rework 210806    
    '''
    await els_home()
    dfp = datapath('ELSScan')
    path, filename = os.path.split(dfp)
    result = str.format("Filename: {0}; Start: {1}; Stop: {2}; Step: {3}; Iterations: {4}; Flashes: {5}; Flash-Mode: {6}; Frequency: {7}; Power: {8}",filename, start, stop, step, iterations, flashes, flash_mode, frequency, power)
    report(result, '.alpha.elsscanah')
    print (filename)
    with open(dfp, 'a') as data:
        data.write(f"pos\tBGRefLo\tBGRefHi\tBGAbsLo\tBGAbsHi\tRefLo\tRefHi\tAbsLo\tAbsHi\n")
        for i in range(start, stop, step):
            pos = i*scale
            await els(pos)
            results = await reftest2ah(iterations, flashes, flash_mode, frequency)
            data.write(f"{pos}\t{results[0]}\t{results[1]}\t{results[2]}\t{results[3]}\t{results[4]}\t{results[5]}\t{results[6]}\t{results[7]}\n")
    await els_home()
    print('ELS_Scan Done\n')
        
async def elsscanb(start=0, stop=3600, step=150, iterations=10, flashes=1, flash_mode=1, frequency=100, power=0.5, scale=0.1):
    '''
    ml 201001:
    rework 210806    
    '''
    await els_home()
    dfp = datapath('ELSScan')
    path, filename = os.path.split(dfp)
    result = str.format("Filename: {0}; Start: {1}; Stop: {2}; Step: {3}; Iterations: {4}; Flashes: {5}; Flash-Mode: {6}; Frequency: {7}; Power: {8}",filename, start, stop, step, iterations, flashes, flash_mode, frequency, power)
    report(result, '.alpha.elsscanb')
    print (filename)
    with open(dfp, 'a') as data:
        data.write(f"pos\tBGRefLo\tBGRefHi\tBGAbsLo\tBGAbsHi\tRefLo\tRefHi\tAbsLo\tAbsHi\n")
        for i in range(start, stop, step):
            pos = i*scale
            await els(pos)
            results = await reftest2b(iterations, flashes, flash_mode, frequency, power)
            data.write(f"{pos}\t{results[0]}\t{results[1]}\t{results[2]}\t{results[3]}\t{results[4]}\t{results[5]}\t{results[6]}\t{results[7]}\n")
    await els_home()
    print('ELS_ScanB Done\n')
        
async def flashscana(start=0, stop=4096, step=10, iterations=10, flashes=1, flash_mode=1, frequency=100):
    '''
    ml 201001:
    rework 210806    
    '''
    await els_home()
    await els_2()
    dfp = datapath('FlashScan')
    path, filename = os.path.split(dfp)
    result = str.format("Filename: {0}; Start: {1}; Stop: {2}; Step: {3}; Iterations: {4}; Flashes: {5}; Flash-Mode: {6}; Frequency: {7}",filename, start, stop, step, iterations, flashes, flash_mode, frequency)
    report(result, '.alpha.flashscana')
    print (filename)
    with open(dfp, 'a') as data:
        data.write(f"pos\tBGRefLo\tBGRefHi\tBGAbsLo\tBGAbsHi\tRefLo\tRefHi\tAbsLo\tAbsHi\n")
        for i in range(start, stop, step):
            await flashsetpower(i)
            results = await reftest2a(iterations, flashes, flash_mode, frequency)
            data.write(f"{i}\t{results[0]}\t{results[1]}\t{results[2]}\t{results[3]}\t{results[4]}\t{results[5]}\t{results[6]}\t{results[7]}\n")
    await els_home()
    print('Flash_ScanA Done\n')
        


async def fmsscan(start=350, stop=400, step=1, iterations=10, flashes=1, flash_mode=1, frequency=100):
    '''
    ml 201001:
    rework 210806    
    '''
    await fms_home()
    dfp = datapath('FMSScan')
    path, filename = os.path.split(dfp)
    result = str.format("Filename: {0}; Start: {1}; Stop: {2}; Step: {3}; Iterations: {4}; Flashes: {5}; Flash-Mode: {6}; Frequency: {7}",filename, start, stop, step, iterations, flashes, flash_mode, frequency)
    report(result, '.alpha.fmsscan')
    print (filename)
    with open(dfp, 'a') as data:
        data.write(f"pos\tBGRefLo\tBGRefHi\tBGAbsLo\tBGAbsHi\tRefLo\tRefHi\tAbsLo\tAbsHi\n")
        for i in range(start, stop, step):
            pos = i*0.1
            await fms(pos)
            results = await reftest2(iterations, flashes, flash_mode, frequency)
            data.write(f"{pos}\t{results[0]}\t{results[1]}\t{results[2]}\t{results[3]}\t{results[4]}\t{results[5]}\t{results[6]}\t{results[7]}\n")
    await fms_home()
    print('FMS_Scan Done\t')
        
async def fmscan(start=0, stop=150, step=1, iterations=10, flashes=1, flash_mode=1, frequency=100):
    '''
    ml 201001:
    rework 210923    
    '''
    await fm_home()
    dfp = datapath('FMScan')
    path, filename = os.path.split(dfp)
    result = str.format("Filename: {0}; Start: {1}; Stop: {2}; Step: {3}; Iterations: {4}; Flashes: {5}; Flash-Mode: {6}; Frequency: {7}",filename, start, stop, step, iterations, flashes, flash_mode, frequency)
    report(result, '.alpha.fmscan')
    print (filename)
    with open(dfp, 'a') as data:
        data.write(f"pos\tBGRefLo\tBGRefHi\tBGAbsLo\tBGAbsHi\tRefLo\tRefHi\tAbsLo\tAbsHi\n")
        for i in range(start, stop, step):
            pos = i*0.1
            await fm(pos)
            results = await reftest2a(iterations, flashes, flash_mode, frequency)
            data.write(f"{pos}\t{results[0]}\t{results[1]}\t{results[2]}\t{results[3]}\t{results[4]}\t{results[5]}\t{results[6]}\t{results[7]}\n")
    await fm_home()
    print('FM_Scan Done\n')
        
async def fmscan2(start=0, stop=150, step=1, iterations=10, flashes=1, flash_mode=1, frequency=100):
    '''
    ml 201001:
    rework 210923    
    '''
    await fm_home()
    dfp = datapath('FMScan')
    path, filename = os.path.split(dfp)
    result = str.format("Filename: {0}; Start: {1}; Stop: {2}; Step: {3}; Iterations: {4}; Flashes: {5}; Flash-Mode: {6}; Frequency: {7}",filename, start, stop, step, iterations, flashes, flash_mode, frequency)
    report(result, '.alpha.fmscan2')
    print (filename)
    with open(dfp, 'a') as data:
        data.write(f"pos\tpmt1_cr\tpmt1_alr\tpmt1_ahr\tpmt1_alo\tpmt1_aho\tpmt2_cr\tpmt2_alr\tpmt2_ahr\tpmt2_alo\tpmt2_aho\tref_alr\tref_ahr\tref_alo\tref_aho\tabs_alr\tabs_ahr\tabs_alo\tabs_aho\n")
        for i in range(start, stop, step):
            pos = i*0.1
            await fm(pos)
            results = await pmttest3a(iterations, flashes, flash_mode, frequency, hv1 = 0.4, hv2 = 0.4, dl1 = 0.25, dl2 = 0.25, pre_cnt_window=10)
            data.write(f"{pos}\t{results[0]}\t{results[1]}\t{results[2]}\t{results[3]}\t{results[4]}\t{results[5]}\t{results[6]}\t{results[7]}\t{results[8]}\t{results[9]}\t{results[10]}\t{results[11]}\t{results[12]}\t{results[13]}\t{results[14]}\t{results[15]}\t{results[16]}\t{results[17]}\n")
    await fm_home()
    print('FM_Scan2 Done\n')
        
async def darkscan(iterations=10):
    '''
    ml 201001:
    darkscan
    '''
    await fms_home()
    await fms(6)
    for i in range(9):
        await reftest(100,iterations,i+1,1)
    for i in range(9):
        await reftest(100,iterations,(i+1)*10,1)
    for i in range(5):
        await reftest(100,iterations,(i+1)*100,1)
    await fms_home()
    print('Done')


async def reftest(frequency, iterations, flashes=1, flash_mode=1, comment='', scale=-1, sindex=3, silent=True, fullresetdelay = 25000, lowresetdelay = 5000):
    '''
    ml 200929:
    
    frequency [Hz]: 0..1000
    '    : : Minimum 3.3V: 2147483647
                 Maximum 2.5V: 1626881551
                 Tested 3.14V: 2043363228
    flash_mode: default: 0 :high speed, low power
                         1 :high power, low speed     
    '''

    delay = 0.1
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    if sindex > 3:
        sindex = 3
    # Limit Freqency for High Power Mode
    if frequency > 500 and flash_mode == 1:
        frequency = 500
    # Set the Flash_Pwr
    curr_value = 3400
    #await hw.flash_setpw(curr_value)
    power = 0.2 #curr_value/4096
    await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampPower, power, timeout=1)

    # Set ontime based on Power Mode 
    if flash_mode == 1:
        ontime = 120000
        #await hw.flash_highpower()
        await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampHighPowerEnable, 0, timeout=1)
    else:
        ontime = 60000
        #await hw.flash_highspeed()
        await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampHighPowerEnable, 1, timeout=1)

    offtime = int(1 / frequency * 1e8 - ontime)
    print(f"Trigger Flashlamp {frequency}Hz, {iterations} Iterations, {flashes} Flashes, Mode {flash_mode}, OnTime {ontime} ticks, OffTime {offtime} ticks\n")
    bar=''
    
    address = 0
    abssequ = [
        0x8C000000,                     # ClearResultBuffer 0 
        0x8C000001,                     # ClearResultBuffer 1 
        0x8C000002,                     # ClearResultBuffer 2 
        0x8C000003,                     # ClearResultBuffer 3 
        0x07000000 | (flashes - 1),     # Loop flashes
        0x7C000000 | (ontime - 1),          # wait for last timer then start ontime timer
        0x02000000 | signals['flash'],      # SetSignal Flash_Trg high
        0x0F012000,                         # SetIntegratorMode(ABS + Ref full Reset)
        0x7C000000 | (lowresetdelay - 1),   # wait for last timer then start lowresetdelay timer
        0x0F01B000,                         # SetIntegratorMode(ABS + Ref low range Reset)
        0x7C000000 | (99),                  # Start Timer for 1µs
        0x03000000 | signals['flash'],      # ResetSignal Flash_Trg low
        0x7C000000 | (3999),                # Start Timer for 40µs
        0x0F024000,                         # SetIntegratorMode(ABS + Ref autorange)
        0x7C000000 | (999),                 # Start Timer for 10µs
        0x0F02D000,                         # SetIntegratorMode(ABS + Ref fixed range)
        0x7C000000,                         # wait for last timer
        0x01000000 | triggers['ref'] | triggers['abs'],      # SetTriggerOutput
        0x84080000,                         # GetAnalogResult (ch=4, rel=0, IgnoreRange=0, IsHiRange=0, add=1, dword=0, addrReg=0, addr=0)
        0x84180001,                         # GetAnalogResult (ch=4, rel=0, IgnoreRange=0, IsHiRange=1, add=1, dword=0, addrReg=0, addr=1)
        0x85080002,                         # GetAnalogResult (ch=5, rel=0, IgnoreRange=0, IsHiRange=0, add=1, dword=0, addrReg=0, addr=2)
        0x85180003,                         # GetAnalogResult (ch=5, rel=0, IgnoreRange=0, IsHiRange=1, add=1, dword=0, addrReg=0, addr=3)
        0x7C000000 | (offtime - 1),         # Wait for ontime timer then start offtime timer
        0x05000000,                     # LoopEnd flashes
        0x00000000,                     # Stop
    ]
    await write_sequence(address, abssequ)
    resultsum = [0] *4
    dfp = datapath('AbsRefTest')
    path, filename = os.path.split(dfp)
    result = str.format("Filename: {0}; Frequency: {1}; Iterations: {2}; Flashes: {3}; Mode: {4}, Ontime: {5}; Offtime: {6}",filename, frequency, iterations, flashes, flash_mode, ontime, offtime)
    report(result, '.alpha.read_ref')
    print (filename)
    with open(dfp, 'a') as data:
        time_stamp = dt.now().strftime('%y%m%d %H:%M:%S')
        if not(silent):
            data.write(f"index\tRefLo\tRefHi\tAbsLo\tAbsHi \n")
        for i in range(iterations):
            try:
                await start_sequence(address)
                results = await get_results(address=0, size=4)
                for index in range(0, len(results)):
                    resultsum[index] = resultsum[index] + results[index]
                if scale < 0:
                    print(f"{i+1:04d}, {results}")
                else:
                    print(f"{i+1:04d}, {bar.ljust(results[sindex]//scale,'#')}")
                #print(f"{i+1}, {results[0]}, {results[1]}, {results[2]}, {results[3]}")
                if not(silent):
                    data.write(f"{i+1:04d}\t{results[0]}\t{results[1]}\t{results[2]}\t{results[3]}\n")
            except BaseException as ex:
                print(f"reftest() failed: {ex}")
            await asyncio.sleep(delay)
        time_stamp = dt.now().strftime('%y%m%d %H:%M:%S')
        data.write(f"\n{time_stamp}, Reference Test: \t{comment}\t{resultsum[0]/iterations}\t{resultsum[1]/iterations}\t{resultsum[2]/iterations}\t{resultsum[3]/iterations}\n")
    print('Done')

async def pmttest(iterations, window=100, scale=-1, sindex=3, comment='', silent=True, hv1=0.4, hv2=0.4, dl1=0.25, dl2=0.25 ):
    '''
    ml 211110:
    
    flash_mode: default: 0 :high speed, low power
                         1 :high power, low speed     ?
    10ns timer tick
    1µs = 100 ticks
    1ms = 100000 ticks
    '''
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit

    delay = 0.1
    await pmt1setdl(dl1)
    await pmt2setdl(dl2)
    await pmt1sethv(hv1)
    await pmt2sethv(hv2)
    await pmt1hvenable()
    await pmt2hvenable()
    await hvgate1on()
    await hvgate2on()

    ## IntegratorModes
    full_reset = 0b010
    low_range_reset = 0b011
    integrate_autorange = 0b100
    integrate_with_fixed_range = 0b101
    integrate_in_low_range = 0b110
    integrate_in_high_range = 0b111

    highresetdelay = 5000   # 50µs
    lowresetdelay = 5000    # 50µs
    meastime = 5000         # 50µs
    delaytime = 5000        # 50µs
    integdelay = 500        # 5µs
    pre_cnt_window = 100    # 1µs
    wscale = 100000 // pre_cnt_window   # 1ms scale for window
    cnt_window = window * wscale
    window_corse, window_fine = divmod(cnt_window, 65536)

    if sindex > 5:
        sindex = 5
    bar=''
    
    op_id = 'pmttest'
    seq_gen = meas_seq_generator()
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=0)   # pmt1, count_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=1)   # pmt1, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=2)   # pmt1, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=3)   # pmt2, count_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=4)   # pmt2, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=5)   # pmt2, analog_high_result
    seq_gen.TimerWaitAndRestart(highresetdelay)
    seq_gen.SetSignals(OutputSignal.Flash)
    seq_gen.SetIntegratorMode(pmt2=full_reset, pmt1=full_reset)
    seq_gen.TimerWaitAndRestart(lowresetdelay)
    seq_gen.SetIntegratorMode(pmt2=low_range_reset, pmt1=low_range_reset)
    seq_gen.TimerWaitAndRestart(delaytime)
    # measurement
    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    seq_gen.SetIntegratorMode(pmt2=integrate_autorange, pmt1=integrate_autorange)
    seq_gen.PulseCounterControl(channel=0, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    seq_gen.PulseCounterControl(channel=1, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    if (window_corse > 0):
        seq_gen.Loop(window_corse)
        seq_gen.Loop(65536)
        seq_gen.TimerWaitAndRestart(pre_cnt_window)
        seq_gen.PulseCounterControl(channel=0, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
        seq_gen.PulseCounterControl(channel=1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
        seq_gen.LoopEnd()
        seq_gen.LoopEnd()
    if (window_fine > 0):
        seq_gen.Loop(window_fine)
        seq_gen.TimerWaitAndRestart(pre_cnt_window)
        seq_gen.PulseCounterControl(channel=0, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
        seq_gen.PulseCounterControl(channel=1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
        seq_gen.LoopEnd()
    seq_gen.TimerWaitAndRestart(integdelay)
    seq_gen.SetIntegratorMode(pmt2=integrate_with_fixed_range, pmt1=integrate_with_fixed_range)
    seq_gen.TimerWait()
    # read out analog signals
    seq_gen.GetPulseCounterResult(channel=0, relative=False, resetCounter=True, cumulative=True, dword=False, addrPos=0, resultPos=0)
    seq_gen.GetPulseCounterResult(channel=1, relative=False, resetCounter=True, cumulative=True, dword=False, addrPos=0, resultPos=3)
    seq_gen.SetTriggerOutput(TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=1)
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=2)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=4)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=5)
    seq_gen.SetIntegratorMode(pmt2=full_reset, pmt1=full_reset)
    seq_gen.Stop(0)

    measurement_unit.ClearOperations()
    measurement_unit.resultAddresses[op_id] = range(0, 6)
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)

    resultsum = [0] *6
    dfp = datapath('PMTTest')
    path, filename = os.path.split(dfp)
    result = str.format("Filename: {0}; Window: {1}; Iterations: {2}",filename, window, iterations)
    report(result, '.alpha.pmttest')
    print (filename)
    with open(dfp, 'a') as data:
        time_stamp = dt.now().strftime('%y%m%d %H:%M:%S')
        if not(silent):
            data.write(f"index\tPMT1Cnt\tPMT1Lo\tPMT1Hi\tPMT2Cnt\tPMT2Lo\tPMT2Hi\n")
        for i in range(iterations):
            try:
                await measurement_unit.ExecuteMeasurement(op_id)
                #await asyncio.sleep(delay)
                results = await measurement_unit.ReadMeasurementValues(op_id)
                for index in range(0, len(results)):
                    resultsum[index] = resultsum[index] + results[index]
                if scale < 0:
                    print(f"{i+1:04d}, {results}")
                else:
                    print(f"{i+1:04d}, {bar.ljust(results[sindex]//scale,'#')}")
                #print(f"{i+1}, {results[0]}, {results[1]}, {results[2]}, {results[3]}")
                if not(silent):
                    data.write(f"{i+1:04d}\t{results[0]}\t{results[1]}\t{results[2]}\t{results[3]}\t{results[4]}\t{results[5]}\n")
            except BaseException as ex:
                print(f"pmttest() failed: {ex}")
            await asyncio.sleep(delay)
        time_stamp = dt.now().strftime('%y%m%d %H:%M:%S')
        data.write(f"\n{time_stamp}, PMT Test: \t{comment}\t{resultsum[0]/iterations}\t{resultsum[1]/iterations}\t{resultsum[2]/iterations}\t{resultsum[3]/iterations}\t{resultsum[4]/iterations}\t{resultsum[5]/iterations}\n")
    print('Done')


async def wellscan3(x=0, y=0, xstart=-225, xstep=5, ystart=-225, ystep=5, repeats = 1):
    '''
    ml 210310:
    
    '''
    steps = 56
    await st_home()
    await st_movetowell(x, y, 0, 0)
    xstop= xstart + xstep* steps
    ystop= ystart + ystep* steps
    print('at Well\n')
    result = str.format("Row: {0}, Column: {1}, XStart: {2}, XStep: {3}, YStart: {4}, YStep: {5}, Repeats: {6}",x , y, xstart, xstep, ystart, ystep, repeats)
    report(result, '.alpha.wellscan3')
    dfp = datapath('WellScan')
    path, filename = os.path.split(dfp)
    report(filename, '.alpha.wellscan3'); #Write to Report
    print (filename)
    with open(dfp, 'a') as data:
        time_stamp = dt.now().strftime('%y%m%d %H:%M:%S')
        data.write(f"\n{time_stamp}, WellScan Test\n")
        data.write(f"Time_stamp1\tTime_stamp2\tTime_stamp3\tj\ti\tyo\txo\tResults\n")
        for j in range(ystart, ystop, ystep):
            yo = j*0.01
            for i in range(xstart, xstop, xstep):
                xo = i*0.01
                await st_movetowell(x, y, xo, yo)
                stpos = await st_getposition()
                xpos = stpos[0]
                ypos = stpos[1]
                result = await reftest2(repeats, flashes=1, flash_mode=1, frequency=100)
                #data.write(f"{j}\t{i}\t{yo}\t{xo}\t{xpos}\t{ypos}\t{result[0]}\t{result[1]}\t{result[2]}\t{result[3]}\n")
                data.write(f"{j}\t{i}\t{yo}\t{xo}\t{xpos}\t{ypos}\t{result[4]}\t{result[5]}\t{result[6]}\t{result[7]}\n")
    await st_home()
    print('Well_Scan Done\n')

async def wellscan3b(x=0, y=0, xstart=-225, xstep=5, ystart=-225, ystep=5, repeats = 1):
    '''
    ml 210310:
    
    '''
    steps = 56
    await st_home()
    await st_movetowell(x, y, 0, 0)
    xstop= xstart + xstep* steps
    ystop= ystart + ystep* steps
    print('at Well\n')
    result = str.format("Row: {0}, Column: {1}, XStart: {2}, XStep: {3}, YStart: {4}, YStep: {5}, Repeats: {6}",x , y, xstart, xstep, ystart, ystep, repeats)
    report(result, '.alpha.wellscan3b')
    dfp = datapath('WellScan')
    path, filename = os.path.split(dfp)
    report(filename, '.alpha.wellscan3b'); #Write to Report
    print (filename)
    with open(dfp, 'a') as data:
        time_stamp = dt.now().strftime('%y%m%d %H:%M:%S')
        data.write(f"\n{time_stamp}, WellScan Test\n")
        data.write(f"j\ti\tyo\txo\tX\tY\tResults\n")
        for j in range(ystart, ystop, ystep):
            yo = j*0.01
            for i in range(xstart, xstop, xstep):
                xo = i*0.01
                await st_movetowell(x, y, xo, yo)
                stpos = await st_getposition()
                xpos = stpos[0]
                ypos = stpos[1]
                results = await pmttest3a(iterations=1, flashes=1, flash_mode=1, frequency=100, hv1 = 0.4, hv2 = 0.4, dl1 = 0.25, dl2 = 0.25, pre_cnt_window=10)
                data.write(f"{j}\t{i}\t{yo}\t{xo}\t{xpos}\t{ypos}\t{results[0]}\t{results[1]}\t{results[2]}\t{results[3]}\t{results[4]}\t{results[5]}\t{results[6]}\t{results[7]}\t{results[8]}\t{results[9]}\t{results[10]}\t{results[11]}\t{results[12]}\t{results[13]}\t{results[14]}\t{results[15]}\t{results[16]}\t{results[17]}\n")
    await st_home()
    print('Well_Scan Done\n')

async def read_ref(frequency, iterations, flashes = 10):
    '''
    ml 210112:
    
    frequency [Hz]: 0..1000
    read reference diode, no flashing
    '''
    #node = VUnits.instance.hal.nodes["EEFNode2"]
    #mf = VUnits.instance.hal.measurementUnit.MeasurementFunctions

    delay = 0.1
    ontime = 50000
    offtime = int(1 / frequency * 1e8 - ontime)
    
    address = 0
    abssequ = [
        0x8C000000,                     # ClearResultBuffer 0 
        0x8C000001,                     # ClearResultBuffer 1 
        0x8C000002,                     # ClearResultBuffer 2 
        0x8C000003,                     # ClearResultBuffer 3 
        0x07000000 | (flashes - 1),     # Loop flashes
        0x02000000 | signals['rstref'] | signals['rstabs'],  # SetSignal Reset RefInt, reset ABSInt
        0x7C000000 | (ontime - 1),      # wait for last timer then start ontime timer
        0x7C000000 | (100),             # Start Timer for 1µs
        0x03000000 | signals['rstref'] | signals['rstabs'],  # ResetSignal Flash_Trg low
        0x7C000000 | (2000),            # Start Timer for 20µs
        0x7C000000 | (100),             # Start Timer for 1µs
        0x01000000 | triggers['ref'] | triggers['abs'],      # SetTriggerOutput
        0x7C000000 | (500),             # Start Timer for 5µs
        0x02000000 | signals['rstref'] | signals['rstabs'],  # SetSignal 
        0x7C000000 | (500),             # Start Timer for 5µs
        0x84080000,                     # GetAnalogResult (ch=4, rel=0, IgnoreRange=0, IsHiRange=0, add=1, dword=0, addrReg=0, addr=0)
        0x84180001,                     # GetAnalogResult (ch=4, rel=0, IgnoreRange=0, IsHiRange=1, add=1, dword=0, addrReg=0, addr=1)
        0x85080002,                     # GetAnalogResult (ch=5, rel=0, IgnoreRange=0, IsHiRange=0, add=1, dword=0, addrReg=0, addr=2)
        0x85180003,                     # GetAnalogResult (ch=5, rel=0, IgnoreRange=0, IsHiRange=1, add=1, dword=0, addrReg=0, addr=3)
        0x7C000000 | (offtime - 1),     # Wait for ontime timer then start offtime timer
        0x05000000,                     # LoopEnd flashes
        0x00000000,                     # Stop
    ]
    await write_sequence(address, abssequ)
    resultsum = [0] *4
    dfp = datapath('PDTest')
    path, filename = os.path.split(dfp)
    result = str.format("Filename: {0}; Frequency: {1}; Iterations: {2}; Flashes: {3}; Ontime: {4}; Offtime: {5}",filename, frequency, iterations, flashes, ontime, offtime)
    report(result, '.alpha.read_ref')
    print (filename)
    with open(dfp, 'a') as data:
        time_stamp = dt.now().strftime('%y%m%d %H:%M:%S')
        data.write(f"index\tRefLo\tRefHi\tAbsLo\tAbsHi \n")
        for i in range(iterations):
            try:
                await start_sequence(address)
                results = await get_results(address=0, size=4)
                for index in range(0, len(results)):
                    resultsum[index] = resultsum[index] + results[index]
                data.write(f"{i+1:04d}\t{results[0]}\t{results[1]}\t{results[2]}\t{results[3]}\n")
            except BaseException as ex:
                print(f"reftest() failed: {ex}")
            await asyncio.sleep(delay)
        time_stamp = dt.now().strftime('%y%m%d %H:%M:%S')
        data.write(f"\n{time_stamp}, Reference Test: \t{resultsum[0]/iterations}\t{resultsum[1]/iterations}\t{resultsum[2]/iterations}\t{resultsum[3]/iterations}\n")
    print('Done')


async def reftest2a(iterations, flashes=1, flash_mode=1, frequency=100, power=0.5):
    '''
    ml 210922:
    
    '''
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit

    ## IntegratorModes
    full_reset = 0b010
    low_range_reset = 0b011
    integrate_autorange = 0b100
    integrate_with_fixed_range = 0b101
    integrate_in_low_range = 0b110
    integrate_in_high_range = 0b111

    lowresetdelay = 5000  # 50µs
    meastime = 5000       # 50µs
    delaytime = 5000      # 50µs
    integdelay = 500      # 5µs
    delay = 0.1
    # Limit Freqency for High Power Mode
    if frequency > 500 and flash_mode == 1:
        frequency = 500

    # Set the Flash_Pwr
    
    await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampPower, power, timeout=1)

    # Set ontime based on Power Mode 
    if flash_mode == 1:
        ontime = 120000
        await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampHighPowerEnable, 0, timeout=1)
    else:
        ontime = 60000
        await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampHighPowerEnable, 1, timeout=1)


    offtime = int(1 / frequency * 1e8 - ontime)

    op_id = 'reftest2a'
    seq_gen = meas_seq_generator()
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=0)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=1)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=2)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=3)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=4)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=5)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=6)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=7)
    seq_gen.Loop(flashes)
    seq_gen.TimerWaitAndRestart(ontime)
    seq_gen.SetSignals(OutputSignal.Flash)
    seq_gen.SetIntegratorMode(abs=full_reset, ref=full_reset)
    seq_gen.TimerWaitAndRestart(lowresetdelay)
    seq_gen.SetIntegratorMode(abs=low_range_reset, ref=low_range_reset)
    seq_gen.TimerWaitAndRestart(delaytime)
    seq_gen.TimerWaitAndRestart(meastime)
    seq_gen.SetIntegratorMode(abs=integrate_in_high_range, ref=integrate_in_high_range)
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=1)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=3)
    seq_gen.TimerWaitAndRestart(meastime)
    seq_gen.SetIntegratorMode(abs=integrate_in_low_range, ref=integrate_in_low_range)
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=0)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=2)
    seq_gen.ResetSignals(OutputSignal.Flash)
    seq_gen.TimerWaitAndRestart(meastime)
    seq_gen.SetIntegratorMode(abs=integrate_autorange, ref=integrate_autorange)
    seq_gen.TimerWaitAndRestart(integdelay)
    seq_gen.SetIntegratorMode(abs=integrate_with_fixed_range, ref=integrate_with_fixed_range)
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=4)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=5)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=6)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=7)
    seq_gen.TimerWaitAndRestart(offtime)
    seq_gen.LoopEnd()
    seq_gen.SetIntegratorMode(abs=full_reset, ref=full_reset)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    measurement_unit.resultAddresses[op_id] = range(0, 8)
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)

    resultsum = [0] *8
    for i in range(iterations):
        try:
            await measurement_unit.ExecuteMeasurement(op_id)
            await asyncio.sleep(delay)
            results = await measurement_unit.ReadMeasurementValues(op_id)
            for index in range(0, len(results)):
                resultsum[index] = resultsum[index] + results[index]
        except BaseException as ex:
                print(f"reftest2a() failed: {ex}")
    return resultsum

async def reftest2al(iterations, flashes=1, flash_mode=1, frequency=100):
    '''
    ml 210922:
    
    '''
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit

    ## IntegratorModes
    full_reset = 0b010
    low_range_reset = 0b011
    integrate_autorange = 0b100
    integrate_with_fixed_range = 0b101
    integrate_in_low_range = 0b110
    integrate_in_high_range = 0b111

    lowresetdelay = 5000  # 50µs
    meastime = 5000       # 50µs
    delaytime = 5000      # 50µs
    integdelay = 500      # 5µs
    delay = 0.1
    power = 0.8
    # Limit Freqency for High Power Mode
    if frequency > 500 and flash_mode == 1:
        frequency = 500

    # Set the Flash_Pwr
    
    await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampPower, power, timeout=1)

    # Set ontime based on Power Mode 
    if flash_mode == 1:
        ontime = 120000
        await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampHighPowerEnable, 0, timeout=1)
    else:
        ontime = 60000
        await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampHighPowerEnable, 1, timeout=1)


    offtime = int(1 / frequency * 1e8 - ontime)

    op_id = 'reftest2al'
    seq_gen = meas_seq_generator()
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=0)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=1)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=2)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=3)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=4)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=5)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=6)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=7)
    seq_gen.Loop(flashes)
    seq_gen.TimerWaitAndRestart(ontime)
    seq_gen.SetSignals(OutputSignal.Flash)
    seq_gen.SetIntegratorMode(abs=full_reset, ref=full_reset)
    seq_gen.TimerWaitAndRestart(lowresetdelay)
    seq_gen.SetIntegratorMode(abs=low_range_reset, ref=low_range_reset)
    seq_gen.TimerWaitAndRestart(delaytime)
    seq_gen.TimerWaitAndRestart(meastime)
    seq_gen.SetIntegratorMode(abs=integrate_in_high_range, ref=integrate_in_high_range)
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=1)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=3)
    seq_gen.TimerWaitAndRestart(meastime)
    seq_gen.SetIntegratorMode(abs=integrate_in_low_range, ref=integrate_in_low_range)
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=0)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=2)
    seq_gen.ResetSignals(OutputSignal.Flash)
    seq_gen.TimerWaitAndRestart(meastime)
    seq_gen.SetIntegratorMode(abs=integrate_in_low_range, ref=integrate_in_low_range)
    seq_gen.TimerWaitAndRestart(integdelay)
    seq_gen.SetIntegratorMode(abs=integrate_with_fixed_range, ref=integrate_with_fixed_range)
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=4)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=5)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=6)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=7)
    seq_gen.TimerWaitAndRestart(offtime)
    seq_gen.LoopEnd()
    seq_gen.SetIntegratorMode(abs=full_reset, ref=full_reset)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    measurement_unit.resultAddresses[op_id] = range(0, 8)
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)

    resultsum = [0] *8
    for i in range(iterations):
        try:
            await measurement_unit.ExecuteMeasurement(op_id)
            await asyncio.sleep(delay)
            results = await measurement_unit.ReadMeasurementValues(op_id)
            for index in range(0, len(results)):
                resultsum[index] = resultsum[index] + results[index]
        except BaseException as ex:
                print(f"reftest2al() failed: {ex}")
    return resultsum

async def reftest2ah(iterations, flashes=1, flash_mode=1, frequency=100):
    '''
    ml 210922:
    
    '''
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit

    ## IntegratorModes
    full_reset = 0b010
    low_range_reset = 0b011
    integrate_autorange = 0b100
    integrate_with_fixed_range = 0b101
    integrate_in_low_range = 0b110
    integrate_in_high_range = 0b111

    lowresetdelay = 5000  # 50µs
    meastime = 5000       # 50µs
    delaytime = 5000      # 50µs
    integdelay = 500      # 5µs
    delay = 0.1
    power = 0.8
    # Limit Freqency for High Power Mode
    if frequency > 500 and flash_mode == 1:
        frequency = 500

    # Set the Flash_Pwr
    
    await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampPower, power, timeout=1)

    # Set ontime based on Power Mode 
    if flash_mode == 1:
        ontime = 120000
        await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampHighPowerEnable, 0, timeout=1)
    else:
        ontime = 60000
        await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampHighPowerEnable, 1, timeout=1)


    offtime = int(1 / frequency * 1e8 - ontime)

    op_id = 'reftest2ah'
    seq_gen = meas_seq_generator()
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=0)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=1)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=2)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=3)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=4)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=5)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=6)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=7)
    seq_gen.Loop(flashes)
    seq_gen.TimerWaitAndRestart(ontime)
    seq_gen.SetSignals(OutputSignal.Flash)
    seq_gen.SetIntegratorMode(abs=full_reset, ref=full_reset)
    seq_gen.TimerWaitAndRestart(lowresetdelay)
    seq_gen.SetIntegratorMode(abs=low_range_reset, ref=low_range_reset)
    seq_gen.TimerWaitAndRestart(delaytime)
    seq_gen.TimerWaitAndRestart(meastime)
    seq_gen.SetIntegratorMode(abs=integrate_in_high_range, ref=integrate_in_high_range)
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=1)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=3)
    seq_gen.TimerWaitAndRestart(meastime)
    seq_gen.SetIntegratorMode(abs=integrate_in_low_range, ref=integrate_in_low_range)
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=0)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=2)
    seq_gen.ResetSignals(OutputSignal.Flash)
    seq_gen.TimerWaitAndRestart(meastime)
    seq_gen.SetIntegratorMode(abs=integrate_in_high_range, ref=integrate_in_high_range)
    seq_gen.TimerWaitAndRestart(integdelay)
    seq_gen.SetIntegratorMode(abs=integrate_with_fixed_range, ref=integrate_with_fixed_range)
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=4)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=5)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=6)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=7)
    seq_gen.TimerWaitAndRestart(offtime)
    seq_gen.LoopEnd()
    seq_gen.SetIntegratorMode(abs=full_reset, ref=full_reset)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    measurement_unit.resultAddresses[op_id] = range(0, 8)
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)

    resultsum = [0] *8
    for i in range(iterations):
        try:
            await measurement_unit.ExecuteMeasurement(op_id)
            await asyncio.sleep(delay)
            results = await measurement_unit.ReadMeasurementValues(op_id)
            for index in range(0, len(results)):
                resultsum[index] = resultsum[index] + results[index]
        except BaseException as ex:
                print(f"reftest2ah() failed: {ex}")
    return resultsum

async def reftest2b(iterations, flashes=1, flash_mode=1, frequency=100, power=0.5):
    '''
    ml 210922:
    no flash
    '''
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit

    ## IntegratorModes
    full_reset = 0b010
    low_range_reset = 0b011
    integrate_autorange = 0b100
    integrate_with_fixed_range = 0b101
    integrate_in_low_range = 0b110
    integrate_in_high_range = 0b111

    lowresetdelay = 5000  # 50µs
    meastime = 5000       # 50µs
    delaytime = 5000      # 50µs
    integdelay = 500      # 5µs
    delay = 0.1
    # Limit Freqency for High Power Mode
    if frequency > 500 and flash_mode == 1:
        frequency = 500

    # Set the Flash_Pwr
    
    await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampPower, power, timeout=1)

    # Set ontime based on Power Mode 
    if flash_mode == 1:
        ontime = 120000
        await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampHighPowerEnable, 0, timeout=1)
    else:
        ontime = 60000
        await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampHighPowerEnable, 1, timeout=1)


    offtime = int(1 / frequency * 1e8 - ontime)

    op_id = 'reftest2b'
    seq_gen = meas_seq_generator()
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=0)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=1)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=2)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=3)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=4)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=5)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=6)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=7)
    seq_gen.Loop(flashes)
    seq_gen.TimerWaitAndRestart(ontime)
    #seq_gen.SetSignals(OutputSignal.Flash)
    seq_gen.SetIntegratorMode(abs=full_reset, ref=full_reset)
    seq_gen.TimerWaitAndRestart(lowresetdelay)
    seq_gen.SetIntegratorMode(abs=low_range_reset, ref=low_range_reset)
    seq_gen.TimerWaitAndRestart(delaytime)
    seq_gen.TimerWaitAndRestart(meastime)
    seq_gen.SetIntegratorMode(abs=integrate_in_high_range, ref=integrate_in_high_range)
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=1)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=3)
    seq_gen.TimerWaitAndRestart(meastime)
    seq_gen.SetIntegratorMode(abs=integrate_in_low_range, ref=integrate_in_low_range)
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=0)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=2)
    #seq_gen.ResetSignals(OutputSignal.Flash)
    seq_gen.TimerWaitAndRestart(meastime)
    seq_gen.SetIntegratorMode(abs=integrate_autorange, ref=integrate_autorange)
    seq_gen.TimerWaitAndRestart(integdelay)
    seq_gen.SetIntegratorMode(abs=integrate_with_fixed_range, ref=integrate_with_fixed_range)
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=4)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=5)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=6)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=7)
    seq_gen.TimerWaitAndRestart(offtime)
    seq_gen.LoopEnd()
    seq_gen.SetIntegratorMode(abs=full_reset, ref=full_reset)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    measurement_unit.resultAddresses[op_id] = range(0, 8)
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)

    resultsum = [0] *8
    for i in range(iterations):
        try:
            await measurement_unit.ExecuteMeasurement(op_id)
            await asyncio.sleep(delay)
            results = await measurement_unit.ReadMeasurementValues(op_id)
            for index in range(0, len(results)):
                resultsum[index] = resultsum[index] + results[index]
        except BaseException as ex:
                print(f"reftest2b() failed: {ex}")
    return resultsum


# 210908  #

async def abssetup(absfilter=5, absfocus=5):
    await st_measpos()
    await st_platetype("96_Opti_Plate")
    await fms_set(absfilter)
    await fm(absfocus)
    result = str.format("Filter Pos.: {0}; Focus Pos.: {1}", absfilter, absfocus) 
    report(result, '.alpha.abssetup')
    return (0)

async def absmeas1b(iterations, flashes = 1, flash_mode = 1, absfilter = 5, absfocus = 5, power = 0.2, delay = 0.2):
    '''
    ml 210906:
    abs measurement using reference- and absorbance-diode
    '''

    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit

    ## IntegratorModes
    full_reset = 0b010
    low_range_reset = 0b011
    integrate_autorange = 0b100
    integrate_with_fixed_range = 0b101
    integrate_in_low_range = 0b110
    integrate_in_high_range = 0b111

    frequency = 100       # 100Hz 
    lowresetdelay = 5000  # 50µs
    meastime = 5000       # 50µs
    # Limit Freqency for High Power Mode
    if frequency > 500 and flash_mode == 1:
        frequency = 500

    # Set the Flash_Pwr
    
    await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampPower, power, timeout=1)

    # Set ontime based on Power Mode 
    if flash_mode == 1:
        ontime = 120000
        await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampHighPowerEnable, 0, timeout=1)
    else:
        ontime = 60000
        await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampHighPowerEnable, 1, timeout=1)


    offtime = int(1 / frequency * 1e8 - ontime)

    await abssetup(absfilter, absfocus)
    await st_home()
    dfp = datapath('ABSTest')
    path, filename = os.path.split(dfp)
    result = str.format("Filename: {0}; Flashes: {1}; Flash Mode: {2}; Flash-Power: {3}; Iterations: {4}",filename, flashes, flash_mode, power, iterations)
    report(result, '.alpha.absmeas1b')
    print (filename)

    op_id = 'absmeastest'
    seq_gen = meas_seq_generator()
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=0)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=1)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=2)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=3)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=4)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=5)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=6)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=7)
    seq_gen.Loop(flashes)
    seq_gen.TimerWaitAndRestart(ontime)
    seq_gen.SetSignals(OutputSignal.Flash)
    seq_gen.SetIntegratorMode(abs=full_reset, ref=full_reset)
    seq_gen.TimerWaitAndRestart(lowresetdelay)
    seq_gen.SetIntegratorMode(abs=low_range_reset, ref=low_range_reset)
    seq_gen.TimerWaitAndRestart(meastime)
    seq_gen.SetIntegratorMode(abs=integrate_in_high_range, ref=integrate_in_high_range)
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=1)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=3)
    seq_gen.TimerWaitAndRestart(meastime)
    seq_gen.SetIntegratorMode(abs=integrate_in_low_range, ref=integrate_in_low_range)
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=0)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=2)
    seq_gen.ResetSignals(OutputSignal.Flash)
    seq_gen.TimerWaitAndRestart(meastime)
    seq_gen.SetIntegratorMode(abs=integrate_autorange, ref=integrate_autorange)
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=4)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=5)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=6)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=7)
    seq_gen.TimerWaitAndRestart(offtime)
    seq_gen.LoopEnd()
    seq_gen.SetIntegratorMode(abs=full_reset, ref=full_reset)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    measurement_unit.resultAddresses[op_id] = range(0, 8)
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)

    with open(dfp, 'a') as data:
        data.write(f"iteration\tbgrl\tbgrh\tbgal\tbgah\tsirl\tsirh\tsial\tsiah\n")
        for i in range(iterations):
            try:
                await measurement_unit.ExecuteMeasurement(op_id)
                await asyncio.sleep(delay)
                results = await measurement_unit.ReadMeasurementValues(op_id)
                data.write(f"{i}\t{results[0]}\t{results[1]}\t{results[2]}\t{results[3]}\t{results[4]}\t{results[5]}\t{results[6]}\t{results[7]}\n")
            except BaseException as ex:
                print(f"absmeas1b() failed: {ex}")
            await asyncio.sleep(delay)

    report('Done', '.alpha.absmeas1b')
    print('absmeas1b Done')
    return (0)

async def absmeas2b(row = 1, iterations = 10, flashes = 1, flash_mode = 1, absfilter = 5, absfocus = 5, meastime = 5000, power = 0.8, delay = 0.2):
    '''
    ml 210907:
    abs measurement using reference- and absorbance-diode
    meastime as parameter, autorange
    '''

    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit

    ## IntegratorModes
    full_reset = 0b010
    low_range_reset = 0b011
    integrate_autorange = 0b100
    integrate_with_fixed_range = 0b101
    integrate_in_low_range = 0b110
    integrate_in_high_range = 0b111

    columns = 12          # 96 well plate
    frequency = 100       # 100Hz 
    lowresetdelay = 5000  # 50µs
    delaytime = 10000     # 100µs
    # Limit Freqency for High Power Mode
    if flash_mode == 1 and frequency > 500:
        frequency = 500

    # Set the Flash_Pwr
    await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampPower, power, timeout=1)

    # Set ontime based on Power Mode 
    if flash_mode == 1:
        ontime = 120000
        await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampHighPowerEnable, 0, timeout=1)
    else:
        ontime = 60000
        await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampHighPowerEnable, 1, timeout=1)

    offtime = int(1 / frequency * 1e8 - ontime)

    await abssetup(absfilter, absfocus)
    await st_home()
    dfp = datapath('ABSTest')
    path, filename = os.path.split(dfp)
    result = str.format("Filename: {0}; Flashes: {1}; Flash Mode: {2}; Meas-Time: {3}; Flash-Power: {4}; Iterations: {5}; DelayTime: {6}",filename, flashes, flash_mode, meastime, power, iterations, delaytime)
    report(result, '.alpha.absmeas2b')
    print (filename)

    op_id = 'absmeastest'
    seq_gen = meas_seq_generator()
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=0)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=1)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=2)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=3)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=4)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=5)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=6)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=7)
    seq_gen.Loop(flashes)
    seq_gen.TimerWaitAndRestart(ontime)
    seq_gen.SetSignals(OutputSignal.Flash)
    seq_gen.SetIntegratorMode(abs=full_reset, ref=full_reset)
    seq_gen.TimerWaitAndRestart(lowresetdelay)
    seq_gen.SetIntegratorMode(abs=low_range_reset, ref=low_range_reset)
    seq_gen.TimerWaitAndRestart(delaytime)
    seq_gen.TimerWaitAndRestart(meastime)
    seq_gen.SetIntegratorMode(abs=integrate_in_high_range, ref=integrate_in_high_range)
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=1)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=3)
    seq_gen.TimerWaitAndRestart(meastime)
    seq_gen.SetIntegratorMode(abs=integrate_in_low_range, ref=integrate_in_low_range)
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=0)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=2)
    seq_gen.ResetSignals(OutputSignal.Flash)
    seq_gen.TimerWaitAndRestart(meastime)
    seq_gen.SetIntegratorMode(abs=integrate_autorange, ref=integrate_autorange)
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=4)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=5)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=6)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=7)
    seq_gen.TimerWaitAndRestart(offtime)
    seq_gen.LoopEnd()
    seq_gen.SetIntegratorMode(abs=full_reset, ref=full_reset)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    measurement_unit.resultAddresses[op_id] = range(0, 8)
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)

    with open(dfp, 'a') as data:
        data.write(f"row\tcolumn\titeration\tbgrl\tbgrh\tbgal\tbgah\tsirl\tsirh\tsial\tsiah\n")
        await st_home()
        for i in range(iterations):
            try:
                await measurement_unit.ExecuteMeasurement(op_id)
                await asyncio.sleep(delay)
                results = await measurement_unit.ReadMeasurementValues(op_id)
                data.write(f"0\t0\t{i}\t{results[0]}\t{results[1]}\t{results[2]}\t{results[3]}\t{results[4]}\t{results[5]}\t{results[6]}\t{results[7]}\n")
            except BaseException as ex:
                print(f"absmeas2b() failed: {ex}")
        for j in range(columns):
            await st_movetowell(j+1, row)
            await asyncio.sleep(delay)
            for i in range(iterations):
                try:
                    await measurement_unit.ExecuteMeasurement(op_id)
                    await asyncio.sleep(delay)
                    results = await measurement_unit.ReadMeasurementValues(op_id)
                    data.write(f"{row}\t{j+1}\t{i}\t{results[0]}\t{results[1]}\t{results[2]}\t{results[3]}\t{results[4]}\t{results[5]}\t{results[6]}\t{results[7]}\n")
                except BaseException as ex:
                    print(f"absmeas2b() failed: {ex}")
                await asyncio.sleep(delay)

    report('Done', '.alpha.absmeas2b')
    print('absmeas2b Done')
    return (0)

async def absmeas2c(row = 1, iterations = 10, flashes = 1, flash_mode = 1, absfilter = 5, absfocus = 5, delaytime = 5000, power = 0.8, delay = 0.2):
    '''
    ml 210907:
    abs measurement using reference- and absorbance-diode
    meastime as parameter, autorange, no flash
    '''

    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit

    ## IntegratorModes
    full_reset = 0b010
    low_range_reset = 0b011
    integrate_autorange = 0b100
    integrate_with_fixed_range = 0b101
    integrate_in_low_range = 0b110
    integrate_in_high_range = 0b111

    columns = 12          # 96 well plate
    frequency = 100       # 100Hz 
    lowresetdelay = 5000  # 50µs
    meastime = 5000       # 50µs
    # Limit Freqency for High Power Mode
    if flash_mode == 1 and frequency > 500:
        frequency = 500

    # Set the Flash_Pwr
    await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampPower, power, timeout=1)

    # Set ontime based on Power Mode 
    if flash_mode == 1:
        ontime = 120000
        await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampHighPowerEnable, 0, timeout=1)
    else:
        ontime = 60000
        await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampHighPowerEnable, 1, timeout=1)

    offtime = int(1 / frequency * 1e8 - ontime)

    await abssetup(absfilter, absfocus)
    await st_home()
    dfp = datapath('ABSTest')
    path, filename = os.path.split(dfp)
    result = str.format("Filename: {0}; Flashes: {1}; Flash Mode: {2}; Meas-Time: {3}; Flash-Power: {4}; Iterations: {5}",filename, flashes, flash_mode, meastime, power, iterations)
    report(result, '.alpha.absmeas2c')
    print (filename)

    op_id = 'absmeastest'
    seq_gen = meas_seq_generator()
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=0)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=1)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=2)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=3)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=4)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=5)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=6)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=7)
    seq_gen.Loop(flashes)
    seq_gen.TimerWaitAndRestart(ontime)
    #seq_gen.SetSignals(OutputSignal.Flash)
    seq_gen.SetIntegratorMode(abs=full_reset, ref=full_reset)
    seq_gen.TimerWaitAndRestart(lowresetdelay)
    seq_gen.SetIntegratorMode(abs=low_range_reset, ref=low_range_reset)
    seq_gen.TimerWaitAndRestart(delaytime)
    seq_gen.TimerWaitAndRestart(meastime)
    seq_gen.SetIntegratorMode(abs=integrate_in_high_range, ref=integrate_in_high_range)
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=1)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=3)
    seq_gen.TimerWaitAndRestart(meastime)
    seq_gen.SetIntegratorMode(abs=integrate_autorange, ref=integrate_autorange)
    #seq_gen.SetIntegratorMode(abs=integrate_in_low_range, ref=integrate_in_low_range)
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=0)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=2)
    #seq_gen.ResetSignals(OutputSignal.Flash)
    seq_gen.TimerWaitAndRestart(meastime)
    seq_gen.SetIntegratorMode(abs=integrate_autorange, ref=integrate_autorange)
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=4)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=5)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=6)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=7)
    seq_gen.TimerWaitAndRestart(offtime)
    seq_gen.LoopEnd()
    seq_gen.SetIntegratorMode(abs=full_reset, ref=full_reset)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    measurement_unit.resultAddresses[op_id] = range(0, 8)
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)

    with open(dfp, 'a') as data:
        data.write(f"row\tcolumn\titeration\tbgrl\tbgrh\tbgal\tbgah\tsirl\tsirh\tsial\tsiah\n")
        await st_home()
        for i in range(iterations):
            try:
                await measurement_unit.ExecuteMeasurement(op_id)
                await asyncio.sleep(delay)
                results = await measurement_unit.ReadMeasurementValues(op_id)
                data.write(f"0\t0\t{i}\t{results[0]}\t{results[1]}\t{results[2]}\t{results[3]}\t{results[4]}\t{results[5]}\t{results[6]}\t{results[7]}\n")
            except BaseException as ex:
                print(f"absmeas2b() failed: {ex}")
        for j in range(columns):
            await st_movetowell(j+1, row)
            await asyncio.sleep(delay)
            for i in range(iterations):
                try:
                    await measurement_unit.ExecuteMeasurement(op_id)
                    await asyncio.sleep(delay)
                    results = await measurement_unit.ReadMeasurementValues(op_id)
                    data.write(f"{row}\t{j+1}\t{i}\t{results[0]}\t{results[1]}\t{results[2]}\t{results[3]}\t{results[4]}\t{results[5]}\t{results[6]}\t{results[7]}\n")
                except BaseException as ex:
                    print(f"absmeas2b() failed: {ex}")
                await asyncio.sleep(delay)

    report('Done', '.alpha.absmeas2c')
    print('absmeas2c Done')
    return (0)

async def absmeas2d(row = 1, iterations = 10, flashes = 1, flash_mode = 1, absfilter = 5, absfocus = 5, meastime = 5000, power = 0.8, delay = 0.2):
    '''
    ml 210907:
    abs measurement using reference- and absorbance-diode
    meastime as parameter, high range only
    '''

    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit

    ## IntegratorModes
    full_reset = 0b010
    low_range_reset = 0b011
    integrate_autorange = 0b100
    integrate_with_fixed_range = 0b101
    integrate_in_low_range = 0b110
    integrate_in_high_range = 0b111

    columns = 12          # 96 well plate
    frequency = 100       # 100Hz 
    lowresetdelay = 5000  # 50µs
    delaytime = 10000     # 100µs
    # Limit Freqency for High Power Mode
    if flash_mode == 1 and frequency > 500:
        frequency = 500

    # Set the Flash_Pwr
    await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampPower, power, timeout=1)

    # Set ontime based on Power Mode 
    if flash_mode == 1:
        ontime = 120000
        await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampHighPowerEnable, 0, timeout=1)
    else:
        ontime = 60000
        await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampHighPowerEnable, 1, timeout=1)

    offtime = int(1 / frequency * 1e8 - ontime)

    await abssetup(absfilter, absfocus)
    await st_home()
    dfp = datapath('ABSTest')
    path, filename = os.path.split(dfp)
    result = str.format("Filename: {0}; Flashes: {1}; Flash Mode: {2}; Meas-Time: {3}; Flash-Power: {4}; Iterations: {5}; DelayTime: {6}",filename, flashes, flash_mode, meastime, power, iterations, delaytime)
    report(result, '.alpha.absmeas2d')
    print (filename)

    op_id = 'absmeastest'
    seq_gen = meas_seq_generator()
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=0)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=1)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=2)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=3)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=4)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=5)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=6)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=7)
    seq_gen.Loop(flashes)
    seq_gen.TimerWaitAndRestart(ontime)
    seq_gen.SetSignals(OutputSignal.Flash)
    seq_gen.SetIntegratorMode(abs=full_reset, ref=full_reset)
    seq_gen.TimerWaitAndRestart(lowresetdelay)
    seq_gen.SetIntegratorMode(abs=low_range_reset, ref=low_range_reset)
    seq_gen.TimerWaitAndRestart(meastime)
    seq_gen.SetIntegratorMode(abs=integrate_in_high_range, ref=integrate_in_high_range)
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=1)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=3)
    seq_gen.TimerWaitAndRestart(meastime)
    seq_gen.SetIntegratorMode(abs=integrate_in_low_range, ref=integrate_in_low_range)
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=0)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=2)
    seq_gen.ResetSignals(OutputSignal.Flash)
    seq_gen.TimerWaitAndRestart(meastime)
    seq_gen.SetIntegratorMode(abs=integrate_in_high_range, ref=integrate_in_high_range)
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=4)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=5)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=6)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=7)
    seq_gen.TimerWaitAndRestart(offtime)
    seq_gen.LoopEnd()
    seq_gen.SetIntegratorMode(abs=full_reset, ref=full_reset)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    measurement_unit.resultAddresses[op_id] = range(0, 8)
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)

    with open(dfp, 'a') as data:
        data.write(f"row\tcolumn\titeration\tbgrl\tbgrh\tbgal\tbgah\tsirl\tsirh\tsial\tsiah\n")
        await st_home()
        for i in range(iterations):
            try:
                await measurement_unit.ExecuteMeasurement(op_id)
                await asyncio.sleep(delay)
                results = await measurement_unit.ReadMeasurementValues(op_id)
                data.write(f"0\t0\t{i}\t{results[0]}\t{results[1]}\t{results[2]}\t{results[3]}\t{results[4]}\t{results[5]}\t{results[6]}\t{results[7]}\n")
            except BaseException as ex:
                print(f"absmeas2d() failed: {ex}")
        for j in range(columns):
            await st_movetowell(j+1, row)
            await asyncio.sleep(delay)
            for i in range(iterations):
                try:
                    await measurement_unit.ExecuteMeasurement(op_id)
                    await asyncio.sleep(delay)
                    results = await measurement_unit.ReadMeasurementValues(op_id)
                    data.write(f"{row}\t{j+1}\t{i}\t{results[0]}\t{results[1]}\t{results[2]}\t{results[3]}\t{results[4]}\t{results[5]}\t{results[6]}\t{results[7]}\n")
                except BaseException as ex:
                    print(f"absmeas2d() failed: {ex}")
                await asyncio.sleep(delay)

    report('Done', '.alpha.absmeas2d')
    print('absmeas2d Done')
    return (0)

async def absmeas2e(row = 1, iterations = 10, flashes = 1, flash_mode = 1, absfilter = 5, absfocus = 5, meastime = 5000, power = 0.8, delay = 0.2):
    '''
    ml 210907:
    abs measurement using reference- and absorbance-diode
    meastime as parameter, high range, no flash
    '''

    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit

    ## IntegratorModes
    full_reset = 0b010
    low_range_reset = 0b011
    integrate_autorange = 0b100
    integrate_with_fixed_range = 0b101
    integrate_in_low_range = 0b110
    integrate_in_high_range = 0b111

    columns = 12            # 96 well plate
    frequency = 100         # 100Hz 
    highresetdelay = 10000  # 100µs
    lowresetdelay = 5000    # 50µs
    delaytime = 10000       # 100µs
    # Limit Freqency for High Power Mode
    if flash_mode == 1 and frequency > 500:
        frequency = 500

    # Set the Flash_Pwr
    await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampPower, power, timeout=1)

    # Set ontime based on Power Mode 
    if flash_mode == 1:
        ontime = 120000
        await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampHighPowerEnable, 0, timeout=1)
    else:
        ontime = 60000
        await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampHighPowerEnable, 1, timeout=1)

    offtime = int(1 / frequency * 1e8 - ontime)

    await abssetup(absfilter, absfocus)
    await st_home()
    dfp = datapath('ABSTest')
    path, filename = os.path.split(dfp)
    result = str.format("Filename: {0}; Flashes: {1}; Flash Mode: {2}; Meas-Time: {3}; Flash-Power: {4}; Iterations: {5}; DelayTime: {6}",filename, flashes, flash_mode, meastime, power, iterations, delaytime)
    report(result, '.alpha.absmeas2e')
    print (filename)

    op_id = 'absmeastest'
    seq_gen = meas_seq_generator()
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=0)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=1)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=2)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=3)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=4)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=5)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=6)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=7)
    seq_gen.TimerWaitAndRestart(highresetdelay)
    seq_gen.SetIntegratorMode(abs=full_reset, ref=full_reset)
    seq_gen.TimerWaitAndRestart(lowresetdelay)
    seq_gen.SetIntegratorMode(abs=low_range_reset, ref=low_range_reset)
    seq_gen.SetIntegratorMode(abs=integrate_in_high_range, ref=integrate_in_high_range)
    seq_gen.Loop(flashes)
    seq_gen.TimerWaitAndRestart(ontime)
    #seq_gen.SetSignals(OutputSignal.Flash)
    seq_gen.TimerWaitAndRestart(offtime)
    #seq_gen.ResetSignals(OutputSignal.Flash)
    seq_gen.LoopEnd()
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=1)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=3)
    seq_gen.TimerWaitAndRestart(highresetdelay)
    seq_gen.SetIntegratorMode(abs=full_reset, ref=full_reset)
    seq_gen.TimerWaitAndRestart(lowresetdelay)
    seq_gen.SetIntegratorMode(abs=low_range_reset, ref=low_range_reset)
    seq_gen.SetIntegratorMode(abs=integrate_in_low_range, ref=integrate_in_low_range)
    seq_gen.Loop(flashes)
    seq_gen.TimerWaitAndRestart(ontime)
    #seq_gen.SetSignals(OutputSignal.Flash)
    seq_gen.TimerWaitAndRestart(offtime)
    #seq_gen.ResetSignals(OutputSignal.Flash)
    seq_gen.LoopEnd()
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=0)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=2)
    seq_gen.TimerWaitAndRestart(highresetdelay)
    seq_gen.SetIntegratorMode(abs=full_reset, ref=full_reset)
    seq_gen.TimerWaitAndRestart(lowresetdelay)
    seq_gen.SetIntegratorMode(abs=low_range_reset, ref=low_range_reset)
    seq_gen.SetIntegratorMode(abs=integrate_in_high_range, ref=integrate_in_high_range)
    #seq_gen.SetIntegratorMode(abs=integrate_autorange, ref=integrate_autorange)
    seq_gen.Loop(flashes)
    seq_gen.TimerWaitAndRestart(ontime)
    seq_gen.SetSignals(OutputSignal.Flash)
    seq_gen.TimerWaitAndRestart(offtime)
    seq_gen.ResetSignals(OutputSignal.Flash)
    seq_gen.LoopEnd()
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=4)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=5)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=6)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=7)
    seq_gen.SetIntegratorMode(abs=full_reset, ref=full_reset)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    measurement_unit.resultAddresses[op_id] = range(0, 8)
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)

    with open(dfp, 'a') as data:
        data.write(f"row\tcolumn\titeration\tbgrl\tbgrh\tbgal\tbgah\tsirl\tsirh\tsial\tsiah\n")
        await st_home()
        for i in range(iterations):
            try:
                await measurement_unit.ExecuteMeasurement(op_id)
                await asyncio.sleep(delay)
                results = await measurement_unit.ReadMeasurementValues(op_id)
                data.write(f"0\t0\t{i}\t{results[0]}\t{results[1]}\t{results[2]}\t{results[3]}\t{results[4]}\t{results[5]}\t{results[6]}\t{results[7]}\n")
            except BaseException as ex:
                print(f"absmeas2e() failed: {ex}")
        for j in range(columns):
            await st_movetowell(j+1, row)
            await asyncio.sleep(delay)
            for i in range(iterations):
                try:
                    await measurement_unit.ExecuteMeasurement(op_id)
                    await asyncio.sleep(delay)
                    results = await measurement_unit.ReadMeasurementValues(op_id)
                    data.write(f"{row}\t{j+1}\t{i}\t{results[0]}\t{results[1]}\t{results[2]}\t{results[3]}\t{results[4]}\t{results[5]}\t{results[6]}\t{results[7]}\n")
                except BaseException as ex:
                    print(f"absmeas2e() failed: {ex}")
                await asyncio.sleep(delay)

    report('Done', '.alpha.absmeas2e')
    print('absmeas2e Done')
    return (0)

async def hellmatest1(iterations = 10, flashes = 1, absfilter = 5, absfocus = 5, meastime = 5000, power = 0.8, delay = 0.2):
    for i in range(1, 9):
        await absmeas2b(i, iterations, flashes, 1, absfilter, absfocus, meastime, power, delay)
    for i in range(1, 9):
        await absmeas2b(i, iterations, flashes, 0, absfilter, absfocus, meastime, power, delay)
    for i in range(1, 9):
        await absmeas2d(i, iterations, flashes, 1, absfilter, absfocus, meastime, power, delay)
    for i in range(1, 9):
        await absmeas2d(i, iterations, flashes, 0, absfilter, absfocus, meastime, power, delay)
    return(0)

async def hellmatest2(nd = 0.0, iterations = 10, flashes = 1, row = 4, absfilter = 5, absfocus = 10, meastime = 5000, power = 0.8, delay = 0.2):
    result = str.format("ND: {0}", nd) 
    report(result, '.alpha.hellmatest2')
    await absmeas2b(row, iterations, flashes, 1, absfilter, absfocus, meastime, power, delay)
    await absmeas2b(row, iterations, flashes, 0, absfilter, absfocus, meastime, power, delay)
    await absmeas2c(row, iterations, flashes, 1, absfilter, absfocus, meastime, power, delay)
    await absmeas2c(row, iterations, flashes, 0, absfilter, absfocus, meastime, power, delay)
    return(0)

async def bldtest(nd = 0.0, start=0, stop=3600, step=150, iterations = 10, flashes = 1, absfilter = 5, absfocus = 10, meastime = 5000, power = 0.8, delay = 0.2):
    await st_home()
    await fms_set(absfilter)
    await fm(absfocus)
    result = str.format("Filter Pos.: {0}; Focus Pos.: {1}; ND: {2}", absfilter, absfocus, nd) 
    report(result, '.alpha.bldtest')
    await bldscana(start, stop, step, iterations, flashes, 1, 100)
    await bldscanb(start, stop, step, iterations, flashes, 1, 100)
    await bldscana(start, stop, step, iterations, flashes, 0, 100)
    await bldscanb(start, stop, step, iterations, flashes, 0, 100)
    return(0)

async def bldtest2(nd = 0.0, start=0, stop=3600, step=150, iterations = 10, flashes = 1, absfilter = 5, absfocus = 10, meastime = 5000, power = 0.8, delay = 0.2):
    await st_home()
    await fms_set(absfilter)
    await fm(absfocus)
    result = str.format("Filter Pos.: {0}; Focus Pos.: {1}; ND: {2}", absfilter, absfocus, nd) 
    report(result, '.alpha.bldtest')
    await bldscana(start, stop, step, iterations, flashes, 1, 100)
    await bldscanal(start, stop, step, iterations, flashes, 1, 100)
    await bldscanah(start, stop, step, iterations, flashes, 1, 100)
    await bldscana(start, stop, step, iterations, flashes, 0, 100)
    await bldscanal(start, stop, step, iterations, flashes, 0, 100)
    await bldscanah(start, stop, step, iterations, flashes, 0, 100)
    return(0)

async def elstest2(nd = 0.0, start=0, stop=3600, step=150, iterations = 10, flashes = 1, absfilter = 5, absfocus = 10, meastime = 5000, power = 0.8, delay = 0.2):
    await st_home()
    await fms_set(absfilter)
    await fm(absfocus)
    result = str.format("Filter Pos.: {0}; Focus Pos.: {1}; ND: {2}", absfilter, absfocus, nd) 
    report(result, '.alpha.elstest')
    await elsscanal(start, stop, step, iterations, flashes, 1, 100, power)
    await elsscanah(start, stop, step, iterations, flashes, 1, 100, power)
    await elsscanal(start, stop, step, iterations, flashes, 0, 100, power)
    await elsscanah(start, stop, step, iterations, flashes, 0, 100, power)
    return(0)

async def fisetup(fifi=3, fifo=8.2):
    await st_measpos()
    #await st_platetype("96_Opti_Plate")
    await st_platetype("384_Opti_Plate")
    await fms_set(fifi)
    await fm(fifo)
    result = str.format("Filter Pos.: {0}; Focus Pos.: {1}", fifi, fifo) 
    report(result, '.alpha.fisetup')
    return (0)

async def fimeas1b(row = 1, cols = 24, iterations = 1, flashes = 1, flash_mode = 1, fifilter = 3, fifocus = 5, hv1 = 0.4, hv2 = 0.5, dl1 = 0.25, dl2 = 0.25, power = 0.8, window = 50):
    '''
    ml 210910:
    FI measurement using reference-diode and pmt
    50µs Window, with Flash
    10ns timer tick
    1µs = 100 ticks
    1ms = 100000 ticks
    '''
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit

    delay = 0.1

    ## IntegratorModes
    full_reset = 0b010
    low_range_reset = 0b011
    integrate_autorange = 0b100
    integrate_with_fixed_range = 0b101
    integrate_in_low_range = 0b110
    integrate_in_high_range = 0b111

    frequency = 100       # 100Hz 
    lowresetdelay = 5000  # 50µs
    meastime = window*100 # in µs
    delaytime = 5000      # 50µs
    integdelay = 500      # 5µs
    pre_cnt_window = 10   # 100ns
    cnt_window = meastime // pre_cnt_window
    #window_corse, window_fine = divmod(cnt_window, 65536)

    # Limit Freqency for High Power Mode
    if frequency > 500 and flash_mode == 1:
        frequency = 500

    # Set the Flash_Pwr
    await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampPower, power, timeout=1)

    # Set ontime based on Power Mode 
    if flash_mode == 1:
        ontime = 120000
        await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampHighPowerEnable, 1, timeout=1)
    else:
        ontime = 60000
        await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampHighPowerEnable, 0, timeout=1)

    offtime = int(1 / frequency * 1e8 - ontime)

    await fisetup(fifilter, fifocus)
    await st_home()

    await pmt1setdl(dl1)
    await pmt2setdl(dl2)
    await pmt1sethv(hv1)
    await pmt2sethv(hv2)
    await pmt1hvenable()
    await pmt2hvenable()
    await hvgate1on()
    await hvgate2on()

    dfp = datapath('FITest')
    path, filename = os.path.split(dfp)
    result = str.format("Filename: {0}; Flashes: {1}; Flash Mode: {2}; Flash-Power: {3}; Iterations: {4}; Window: {5}",filename, flashes, flash_mode, power, iterations, window)
    report(result, '.alpha.fimeas1b')
    result = str.format("Row: {0}; HV1: {1}; HV2: {2}; DL1: {3}; DL2: {4}; CntWindow: {5};PreCntWindow: {6}", row, hv1, hv2, dl1, dl2, cnt_window, pre_cnt_window)
    report(result, '.alpha.fimeas1b')
    print (filename)

    op_id = 'fimeastest'
    seq_gen = meas_seq_generator()
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=0)   # pmt1, count_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=1)   # pmt1, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=2)   # pmt1, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=3)   # pmt1, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=4)   # pmt1, analog_high_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=5)   # pmt2, count_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=6)   # pmt2, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=7)   # pmt2, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=8)   # pmt2, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=9)   # pmt2, analog_high_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=10)  # ref, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=11)  # ref, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=12)  # ref, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=13)  # ref, analog_high_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=14)  # abs, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=15)  # abs, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=16)  # abs, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=17)  # abs, analog_high_offset
    seq_gen.Loop(flashes)
    seq_gen.TimerWaitAndRestart(ontime)
    seq_gen.SetSignals(OutputSignal.Flash)
    seq_gen.SetIntegratorMode(abs=full_reset, ref=full_reset, pmt2=full_reset, pmt1=full_reset)
    seq_gen.TimerWaitAndRestart(lowresetdelay)
    seq_gen.SetIntegratorMode(abs=low_range_reset, ref=low_range_reset, pmt2=low_range_reset, pmt1=low_range_reset)
    seq_gen.TimerWaitAndRestart(delaytime)
    # high range offset
    seq_gen.TimerWaitAndRestart(meastime)
    seq_gen.SetIntegratorMode(abs=integrate_in_high_range, ref=integrate_in_high_range, pmt2=integrate_in_high_range, pmt1=integrate_in_high_range)
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)
    # channels: pmt1= 0; pmt2= 1; ref= 4; abs= 5
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=4)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=9)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=13)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=17)
    # low range offset
    seq_gen.TimerWaitAndRestart(meastime)
    seq_gen.SetIntegratorMode(abs=integrate_in_low_range, ref=integrate_in_low_range, pmt2=integrate_in_low_range, pmt1=integrate_in_low_range)
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=3)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=8)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=12)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=16)
    seq_gen.ResetSignals(OutputSignal.Flash)
    # measurement
    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    seq_gen.SetIntegratorMode(abs=integrate_autorange, ref=integrate_autorange, pmt2=integrate_autorange, pmt1=integrate_autorange)
    seq_gen.PulseCounterControl(channel=0, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    seq_gen.PulseCounterControl(channel=1, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    seq_gen.Loop(cnt_window)
    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    seq_gen.PulseCounterControl(channel=0, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
    seq_gen.PulseCounterControl(channel=1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
    seq_gen.LoopEnd()
    seq_gen.TimerWaitAndRestart(integdelay)
    seq_gen.SetIntegratorMode(abs=integrate_with_fixed_range, ref=integrate_with_fixed_range, pmt2=integrate_with_fixed_range, pmt1=integrate_with_fixed_range)
    seq_gen.TimerWait()
    # read out analog signals
    seq_gen.GetPulseCounterResult(channel=0, relative=False, resetCounter=True, cumulative=True, dword=False, addrPos=0, resultPos=0)
    seq_gen.GetPulseCounterResult(channel=1, relative=False, resetCounter=True, cumulative=True, dword=False, addrPos=0, resultPos=5)
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=1)
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=2)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=6)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=7)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=10)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=11)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=14)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=15)
    seq_gen.TimerWaitAndRestart(offtime)
    seq_gen.LoopEnd()
    seq_gen.SetIntegratorMode(abs=full_reset, ref=full_reset, pmt2=full_reset, pmt1=full_reset)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    measurement_unit.resultAddresses[op_id] = range(0, 18)
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)

    with open(dfp, 'a') as data:
        data.write(f"Colmn\tpmt1_cr\tpmt1_alr\tpmt1_ahr\tpmt1_alo\tpmt1_aho\tpmt2_cr\tpmt2_alr\tpmt2_ahr\tpmt2_alo\tpmt2_aho\tref_alr\tref_ahr\tref_alo\tref_aho\tabs_alr\tabs_ahr\tabs_alo\tabs_aho\n")
        for j in range(cols):
            await st_movetowell(j+1, row)
            for i in range(iterations):
                try:
                    await measurement_unit.ExecuteMeasurement(op_id)
                    await asyncio.sleep(delay)
                    results = await measurement_unit.ReadMeasurementValues(op_id)
                    data.write(f"{j+1}\t{results[0]}\t{results[1]}\t{results[2]}\t{results[3]}\t{results[4]}\t{results[5]}\t{results[6]}\t{results[7]}\t{results[8]}\t{results[9]}\t{results[10]}\t{results[11]}\t{results[12]}\t{results[13]}\t{results[14]}\t{results[15]}\t{results[16]}\t{results[17]}\n")
                except BaseException as ex:
                    print(f"fimeas1b() failed: {ex}")
                await asyncio.sleep(delay)

    await hvgate1off()
    await hvgate2off()
    await pmt1hvdisable()
    await pmt2hvdisable()

    report('Done', '.alpha.fimeas1b')
    print('fimeas1b Done')
    return (0)

async def fimeas1c(iterations, flashes = 1, flash_mode = 1, fifilter = 3, fifocus = 5, hv1 = 0.4, hv2 = 0.5, dl1 = 0.25, dl2 = 0.25, power = 0.8, delay = 0.2):
    '''
    ml 210910:
    FI measurement using reference-diode and pmt
    '''

    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit

    ## IntegratorModes
    full_reset = 0b010
    low_range_reset = 0b011
    integrate_autorange = 0b100
    integrate_with_fixed_range = 0b101
    integrate_in_low_range = 0b110
    integrate_in_high_range = 0b111

    frequency = 100       # 100Hz 
    lowresetdelay = 5000  # 50µs
    meastime = 50       # 50µs
    # Limit Freqency for High Power Mode
    if flash_mode == 1 and frequency > 500:
        frequency = 500

    # Set the Flash_Pwr
    await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampPower, power, timeout=1)

    # Set ontime based on Power Mode 
    if flash_mode == 1:
        ontime = 120000
        await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampHighPowerEnable, 0, timeout=1)
    else:
        ontime = 60000
        await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampHighPowerEnable, 1, timeout=1)

    offtime = int(1 / frequency * 1e8 - ontime)

    await fisetup(fifilter, fifocus)
    await st_home()

    await pmt1setdl(dl1)
    await pmt2setdl(dl2)
    await pmt1sethv(hv1)
    await pmt2sethv(hv2)
    await pmt1hvenable()
    await pmt2hvenable()
    await hvgate1on()
    await hvgate2on()

    dfp = datapath('FITest')
    path, filename = os.path.split(dfp)
    result = str.format("Filename: {0}; Flashes: {1}; Flash Mode: {2}; Iterations: {3}",filename, flashes, flash_mode, iterations)
    report(result, '.alpha.fimeas1c')
    print (filename)

    op_id = 'fimeastest'
    seq_gen = meas_seq_generator()
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=0)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=1)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=2)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=3)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=4)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=5)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=6)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=7)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=8)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=9) 
    seq_gen.Loop(flashes)
    seq_gen.TimerWaitAndRestart(ontime)
    seq_gen.SetSignals(OutputSignal.Flash)
    seq_gen.SetIntegratorMode(pmt1=full_reset, ref=full_reset)
    seq_gen.TimerWaitAndRestart(lowresetdelay)
    seq_gen.SetIntegratorMode(pmt1=low_range_reset, ref=low_range_reset)
    seq_gen.SetSignals(OutputSignal.InputGatePMT1)
    seq_gen.TimerWaitAndRestart(100)
    seq_gen.SetIntegratorMode(pmt1=integrate_in_high_range, ref=integrate_in_high_range)
    seq_gen.Loop(meastime)
    seq_gen.TimerWaitAndRestart(100)
    seq_gen.LoopEnd()
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SamplePMT1)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=1)
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=3)
    seq_gen.TimerWaitAndRestart(100)
    seq_gen.SetIntegratorMode(pmt1=integrate_in_low_range, ref=integrate_in_low_range)
    seq_gen.PulseCounterControl(channel=0, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    seq_gen.Loop(meastime)
    seq_gen.TimerWaitAndRestart(100)
    seq_gen.PulseCounterControl(channel=0, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
    seq_gen.LoopEnd()
    seq_gen.TimerWait()
    seq_gen.GetPulseCounterResult(channel=0, relative=False, resetCounter=True, cumulative=False, dword=False, addrPos=0, resultPos=4)
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SamplePMT1)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=0)
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=2)
    seq_gen.ResetSignals(OutputSignal.Flash)
    seq_gen.TimerWaitAndRestart(100)
    seq_gen.SetIntegratorMode(pmt1=integrate_autorange, ref=integrate_autorange)
    seq_gen.PulseCounterControl(channel=0, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    seq_gen.Loop(meastime)
    seq_gen.TimerWaitAndRestart(100)
    seq_gen.PulseCounterControl(channel=0, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
    seq_gen.LoopEnd()
    seq_gen.TimerWait()
    seq_gen.GetPulseCounterResult(channel=0, relative=False, resetCounter=True, cumulative=True, dword=False, addrPos=0, resultPos=9)
    seq_gen.ResetSignals(OutputSignal.InputGatePMT1)
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SamplePMT1)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=5)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=6)
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=7)
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=8)
    seq_gen.TimerWaitAndRestart(offtime)
    seq_gen.LoopEnd()
    seq_gen.SetIntegratorMode(pmt1=full_reset, ref=full_reset)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    measurement_unit.resultAddresses[op_id] = range(0, 10)
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)

    with open(dfp, 'a') as data:
        data.write(f"iter\tbgrl\tbgrh\tbgp1l\tbgp1h\tbgp1c\tsirl\tsirh\tsip1l\tsip1h\tsip1c\n")
        for i in range(iterations):
            try:
                await measurement_unit.ExecuteMeasurement(op_id)
                await asyncio.sleep(delay)
                results = await measurement_unit.ReadMeasurementValues(op_id)
                data.write(f"{i}\t{results[0]}\t{results[1]}\t{results[2]}\t{results[3]}\t{results[4]}\t{results[5]}\t{results[6]}\t{results[7]}\t{results[8]}\t{results[9]}\n")
            except BaseException as ex:
                print(f"fimeas1c() failed: {ex}")
            await asyncio.sleep(delay)

    await hvgate1off()
    await hvgate2off()
    await pmt1hvdisable()
    await pmt2hvdisable()
    report('Done', '.alpha.fimeas1c')
    print('fimeas1c Done')
    return (0)

async def pmteval1():
    await st_home()
    report('st_home', '.alpha.pmteval1')
    await asx_close()
    report('asx_close', '.alpha.pmteval1')
    await fisetup(3, 8.2)
    await st_movetowell(4, 4)
    report('st_movetowell(4, 4)', '.alpha.pmteval1')
    await dlscan(dlstart=0, dlstop=255, dlstep=1, window=100, pre_cnt_window=10)
    return(0)

async def pmteval2(dl=70):
    await st_home()
    report('st_home', '.alpha.pmteval2')
    await asx_close()
    report('asx_close', '.alpha.pmteval2')
    await fisetup(3, 8.2)
    await st_movetowell(4, 4)
    report('st_movetowell(4, 4)', '.alpha.pmteval2')
    await hvscan(dl, hvstart=0, hvstop=188, hvstep=1, flashes=100, flash_mode = 0, pre_cnt_window=10)
    await asx(1.0)
    report('asx(1.0)', '.alpha.pmteval2')
    await hvscan(dl, hvstart=0, hvstop=188, hvstep=1, flashes=100, flash_mode = 0, pre_cnt_window=10)
    await asx(1.3)
    report('asx(1.3)', '.alpha.pmteval2')
    await hvscan(dl, hvstart=0, hvstop=188, hvstep=1, flashes=100, flash_mode = 0, pre_cnt_window=10)
    await asx(1.35)
    report('asx(1.35)', '.alpha.pmteval2')
    await hvscan(dl, hvstart=0, hvstop=188, hvstep=1, flashes=100, flash_mode = 0, pre_cnt_window=10)
    await asx(1.4)
    report('asx(1.4)', '.alpha.pmteval2')
    await hvscan(dl, hvstart=0, hvstop=188, hvstep=1, flashes=100, flash_mode = 0, pre_cnt_window=10)
    await asx(1.45)
    report('asx(1.45)', '.alpha.pmteval2')
    await hvscan(dl, hvstart=0, hvstop=188, hvstep=1, flashes=100, flash_mode = 0, pre_cnt_window=10)
    await asx(1.5)
    report('asx(1.5)', '.alpha.pmteval2')
    await hvscan(dl, hvstart=0, hvstop=188, hvstep=1, flashes=100, flash_mode = 0, pre_cnt_window=10)
    await asx_close()
    report('asx_close', '.alpha.pmteval2')
    return(0)

async def pmteval3(dl1 = 0.25, dl2 = 0.25, hv1 = 0.4, hv2 = 0.4):
    start=500
    stop=4400
    step=10
    iterations=1
    flashes=100
    flash_mode=0
    frequency=100
    pre_cnt_window=10
    await st_home()
    report('st_home', '.alpha.pmteval3')
    await asx_close()
    report('asx_close', '.alpha.pmteval3')
    await fisetup(3, 8.2)
    await st_movetowell(4, 4)
    report('st_movetowell(4, 4)', '.alpha.pmteval3')
    await as1scana(start, stop, step, iterations, flashes, flash_mode, frequency, hv1, hv2, dl1, dl2, pre_cnt_window )
    await as1scanal(start, stop, step, iterations, flashes, flash_mode, frequency, hv1, hv2, dl1, dl2, pre_cnt_window )
    await as1scanah(start, stop, step, iterations, flashes, flash_mode, frequency, hv1, hv2, dl1, dl2, pre_cnt_window )
    await asx_close()
    report('asx_close', '.alpha.pmteval3')
    return(0)

async def fieval1(dl1 = 0.25, dl2 = 0.25, hv1 = 0.4, hv2 = 0.4):
    await st_home()
    report('st_home', '.alpha.fieval1')
    await asx_close()
    report('asx_close', '.alpha.fieval1')
    await fisetup(3, 8.2)
    await st_movetowell(4, 4)
    report('st_movetowell(4, 4)', '.alpha.fieval1')
    await fimeas1b(row = 4, cols = 24, iterations = 1, flashes = 100, flash_mode = 0, fifilter = 3, fifocus = 8.2, hv1 = 0.4, hv2 = 0.4, dl1 = 0.25, dl2 = 0.25, power = 0.3, window = 50)
    await fimeas1b(row = 4, cols = 24, iterations = 1, flashes = 100, flash_mode = 1, fifilter = 3, fifocus = 8.2, hv1 = 0.4, hv2 = 0.4, dl1 = 0.25, dl2 = 0.25, power = 0.3, window = 50)
    await asx_1()
    report('asx_1', '.alpha.fieval1')
    await fimeas1b(row = 4, cols = 24, iterations = 1, flashes = 100, flash_mode = 0, fifilter = 3, fifocus = 8.2, hv1 = 0.4, hv2 = 0.4, dl1 = 0.25, dl2 = 0.25, power = 0.3, window = 50)
    await fimeas1b(row = 4, cols = 24, iterations = 1, flashes = 100, flash_mode = 1, fifilter = 3, fifocus = 8.2, hv1 = 0.4, hv2 = 0.4, dl1 = 0.25, dl2 = 0.25, power = 0.3, window = 50)
    await asx_3()
    report('asx_3', '.alpha.fieval1')
    await fimeas1b(row = 4, cols = 24, iterations = 1, flashes = 100, flash_mode = 0, fifilter = 3, fifocus = 8.2, hv1 = 0.4, hv2 = 0.4, dl1 = 0.25, dl2 = 0.25, power = 0.3, window = 50)
    await fimeas1b(row = 4, cols = 24, iterations = 1, flashes = 100, flash_mode = 1, fifilter = 3, fifocus = 8.2, hv1 = 0.4, hv2 = 0.4, dl1 = 0.25, dl2 = 0.25, power = 0.3, window = 50)
    await asx_close()
    report('asx_close', '.alpha.fieval1')
    return(0)

async def flash_excitation_scan1(start_us=0.0, window_us=1.0, window_count=100, flash_count=1, flash_power=0.5, high_power=1, gatedelay=1):
    if (start_us < 0.0) or (start_us > 671088.64):
        raise ValueError(f"start_us must be in the range [0.0, 671088.64] us")
    if (window_us < 0.1) or (window_us > 671088.64):
        raise ValueError(f"step_us must be in the range [0.1, 671088.64] us")
    if (window_count < 1) or (window_count > 4096):
        raise ValueError(f"window_count must be in the range [1, 4096]")
    if (flash_count < 1) or (flash_count > 65536):
        raise ValueError(f"flash_count must be in the range [1, 65536]")
    if (flash_power < 0.0) or (flash_power > 1.0):
        raise ValueError(f"flash_power must be in the range [0.0, 1.0]")
    if (high_power != 0) and (high_power != 1) and (high_power != None):
        raise ValueError(f"high_power must be in 0 or 1")

    meas_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit

    if flash_power is not None:
        await meas_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampPower, flash_power, timeout=1)
    if high_power is not None:
        await meas_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampHighPowerEnable, high_power, timeout=1)
    else:
        high_power = (await meas_unit.MeasurementFunctions.GetParameter(MeasurementParameter.FlashLampHighPowerEnable, timeout=1))[0]

    if not high_power:
        arming = 50000   #  500 us arming time
    else:
        arming = 120000  # 1200 us arming time
    delay = round(start_us * 100)
    window = round(window_us * 100)
    # 250 Hz flash frequency:
    loop_delay = 400000 - arming - delay - window * window_count

    op_id = 'flash_excitation_scan'
    seq_gen = meas_seq_generator()
    # PMT high voltage gate on
    seq_gen.TimerWaitAndRestart(gatedelay)
    seq_gen.SetSignals(OutputSignal.HVGatePMT1)
    seq_gen.TimerWait()
    # Clear the result buffer
    seq_gen.SetAddrReg(relative=False, dataNotAddrSrc=False, sign=False, stackNotRegSrc=False, srcReg=0, dstReg=0, addr=0)
    seq_gen.Loop(window_count)
    seq_gen.ClearResultBuffer(relative=True, dword=False, addrReg=0, addr=0)
    seq_gen.SetAddrReg(relative=True, dataNotAddrSrc=False, sign=False, stackNotRegSrc=False, srcReg=0, dstReg=0, addr=1)
    seq_gen.LoopEnd()

    seq_gen.Loop(flash_count)
    # Arm the flash lamp
    seq_gen.TimerWaitAndRestart(arming)
    seq_gen.SetSignals(OutputSignal.Flash)
    # Trigger the flash
    seq_gen.TimerWaitAndRestart(delay)
    seq_gen.ResetSignals(OutputSignal.Flash)
    # Start the scan with PMT1
    seq_gen.TimerWaitAndRestart(window)
    seq_gen.PulseCounterControl(channel=0, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    seq_gen.SetAddrReg(relative=False, dataNotAddrSrc=False, sign=False, stackNotRegSrc=False, srcReg=0, dstReg=0, addr=0)
    # Loop to measure each window and save the result
    seq_gen.Loop(window_count)
    seq_gen.TimerWaitAndRestart(window)
    seq_gen.PulseCounterControl(channel=0, cumulative=False, resetCounter=False, resetPresetCounter=True, correctionOn=True)
    seq_gen.GetPulseCounterResult(channel=0, relative=True, resetCounter=True, cumulative=True, dword=False, addrPos=0, resultPos=0)
    seq_gen.SetAddrReg(relative=True, dataNotAddrSrc=False, sign=False, stackNotRegSrc=False, srcReg=0, dstReg=0, addr=1)
    seq_gen.LoopEnd()
    if (loop_delay > 0):
        seq_gen.TimerWaitAndRestart(loop_delay)
    seq_gen.LoopEnd()
    seq_gen.ResetSignals(OutputSignal.HVGatePMT1)
    seq_gen.Stop(0)
    
    meas_unit.ClearOperations()
    meas_unit.resultAddresses[op_id] = range(0, window_count)
    await meas_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
    await meas_unit.ExecuteMeasurement(op_id)
    results = await meas_unit.ReadMeasurementValues(op_id)

    return results

async def flash_excitation_scan2(start_us=0.0, window_us=1.0, window_count=100, flash_count=1, flash_power=0.5, high_power=1, gatedelay=1):
    if (start_us < 0.0) or (start_us > 671088.64):
        raise ValueError(f"start_us must be in the range [0.0, 671088.64] us")
    if (window_us < 0.1) or (window_us > 671088.64):
        raise ValueError(f"step_us must be in the range [0.1, 671088.64] us")
    if (window_count < 1) or (window_count > 4096):
        raise ValueError(f"window_count must be in the range [1, 4096]")
    if (flash_count < 1) or (flash_count > 65536):
        raise ValueError(f"flash_count must be in the range [1, 65536]")
    if (flash_power < 0.0) or (flash_power > 1.0):
        raise ValueError(f"flash_power must be in the range [0.0, 1.0]")
    if (high_power != 0) and (high_power != 1) and (high_power != None):
        raise ValueError(f"high_power must be in 0 or 1")

    meas_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit

    if flash_power is not None:
        await meas_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampPower, flash_power, timeout=1)
    if high_power is not None:
        await meas_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampHighPowerEnable, high_power, timeout=1)
    else:
        high_power = (await meas_unit.MeasurementFunctions.GetParameter(MeasurementParameter.FlashLampHighPowerEnable, timeout=1))[0]

    if not high_power:
        arming = 50000   #  500 us arming time
    else:
        arming = 120000  # 1200 us arming time
    delay = round(start_us * 100)
    window = round(window_us * 100)
    # 250 Hz flash frequency:
    loop_delay = 400000 - arming - delay - window * window_count

    op_id = 'flash_excitation_scan'
    seq_gen = meas_seq_generator()
    # PMT high voltage gate on
    seq_gen.TimerWaitAndRestart(gatedelay)
    seq_gen.SetSignals(OutputSignal.HVGatePMT2)
    seq_gen.TimerWait()
    # Clear the result buffer
    seq_gen.SetAddrReg(relative=False, dataNotAddrSrc=False, sign=False, stackNotRegSrc=False, srcReg=0, dstReg=0, addr=0)
    seq_gen.Loop(window_count)
    seq_gen.ClearResultBuffer(relative=True, dword=False, addrReg=0, addr=0)
    seq_gen.SetAddrReg(relative=True, dataNotAddrSrc=False, sign=False, stackNotRegSrc=False, srcReg=0, dstReg=0, addr=1)
    seq_gen.LoopEnd()

    seq_gen.Loop(flash_count)
    # Arm the flash lamp
    seq_gen.TimerWaitAndRestart(arming)
    seq_gen.SetSignals(OutputSignal.Flash)
    # Trigger the flash
    seq_gen.TimerWaitAndRestart(delay)
    seq_gen.ResetSignals(OutputSignal.Flash)
    # Start the scan with PMT1
    seq_gen.TimerWaitAndRestart(window)
    seq_gen.PulseCounterControl(channel=1, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    seq_gen.SetAddrReg(relative=False, dataNotAddrSrc=False, sign=False, stackNotRegSrc=False, srcReg=0, dstReg=0, addr=0)
    # Loop to measure each window and save the result
    seq_gen.Loop(window_count)
    seq_gen.TimerWaitAndRestart(window)
    seq_gen.PulseCounterControl(channel=1, cumulative=False, resetCounter=False, resetPresetCounter=True, correctionOn=True)
    seq_gen.GetPulseCounterResult(channel=1, relative=True, resetCounter=True, cumulative=True, dword=False, addrPos=0, resultPos=0)
    seq_gen.SetAddrReg(relative=True, dataNotAddrSrc=False, sign=False, stackNotRegSrc=False, srcReg=0, dstReg=0, addr=1)
    seq_gen.LoopEnd()
    if (loop_delay > 0):
        seq_gen.TimerWaitAndRestart(loop_delay)
    seq_gen.LoopEnd()
    seq_gen.ResetSignals(OutputSignal.HVGatePMT2)
    seq_gen.Stop(0)
    
    meas_unit.ClearOperations()
    meas_unit.resultAddresses[op_id] = range(0, window_count)
    await meas_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
    await meas_unit.ExecuteMeasurement(op_id)
    results = await meas_unit.ReadMeasurementValues(op_id)

    return results

async def flash(as_pos=1.0, dl1=0.3, dl2=0.3, hv1=0.5, hv2=0.5, start_us=0.0, window_us=1.0, window_count=100, flash_count=1, flash_power=0.5, high_power=1, gatedelay=1):
    await pmt1setdl(dl1)
    await pmt2setdl(dl2)
    await pmt1sethv(hv1)
    await pmt2sethv(hv2)
    await pmt1hvenable()
    await pmt2hvenable()
    #await hvgate1on()
    #await hvgate2on()
    await asx(as_pos)

    dfp = datapath('FlashDecay')
    path, filename = os.path.split(dfp)
    result = str.format("Filename: {0}; Flashes: {1}; Flash Mode: {2}; Flash-Power: {3}; Start: {4}; Window: {5}",filename, flash_count, high_power, flash_power, start_us, window_us)
    report(result, '.alpha.flash')
    result = str.format("HV1: {0}; HV2: {1}; DL1: {2}; DL2: {3}; Aperture-Pos.: {4}; GateDelay: {5}", hv1, hv2, dl1, dl2, as_pos, gatedelay)
    report(result, '.alpha.flash')
    print (filename)

    results1 = await flash_excitation_scan1(start_us, window_us, window_count, flash_count, flash_power, high_power, gatedelay)
    results2 = await flash_excitation_scan2(start_us, window_us, window_count, flash_count, flash_power, high_power, gatedelay)
    with open(dfp, 'a') as data:
        data.write(f"Index\tTime\tCounts1\tCounts2\n")
        for i in range(window_count):
            data.write(f"{i}\t{start_us+i*window_us}\t{results1[i]}\t{results2[i]}\n")

    await asx_close()
    #await hvgate1off()
    #await hvgate2off()
    await pmt1hvdisable()
    await pmt2hvdisable()

    report('Done', '.alpha.flash')
    print('flash Done')
    return (0)
    
async def flashmap(as_pos=1.0, dlstart=56, dlstop=127, dlstep=1, hv1=0.5, hv2=0.5, start_us=0.0, window_us=1.0, window_count=100, flash_count=1, flash_power=0.5, high_power=1, gatedelay=1):
    delay = 0.1
    await pmt1setdl(dlstart/256)
    await pmt2setdl(dlstart/256)
    await pmt1sethv(hv1)
    await pmt2sethv(hv2)
    await pmt1hvenable()
    await pmt2hvenable()
    #await hvgate1on()
    #await hvgate2on()
    await asx(as_pos)
    await asyncio.sleep(delay)

    print ("Channel 1")
    dfp = datapath('FlashDecay')
    path, filename = os.path.split(dfp)
    result = str.format("Filename: {0}; Flashes: {1}; Flash Mode: {2}; Flash-Power: {3}; Start: {4}; Window: {5}",filename, flash_count, high_power, flash_power, start_us, window_us)
    report(result, '.alpha.flashmap')
    result = str.format("HV1: {0}; HV2: {1}; DL1: {2}; DL2: {3}; Aperture-Pos.: {4}; GateDelay: {5}", hv1, hv2, dlstart, dlstop, as_pos, gatedelay)
    report(result, '.alpha.flashmap')
    print (filename)

    arr=[]
    di = []
    for i in range(window_count):
        di.append(start_us+i*window_us)
    for dl in range(dlstart, dlstop+dlstep, dlstep):
        await pmt1setdl(dl/256)
        await pmt2setdl(dl/256)
        await asyncio.sleep(delay)
        results = await flash_excitation_scan1(start_us, window_us, window_count, flash_count, flash_power, high_power, gatedelay)
        arr.append(results)
    await asx_close()
    await hvgate1off()
    await hvgate2off()
    await pmt1hvdisable()
    await pmt2hvdisable()
    with open(dfp, 'a') as data:
        for j in range(len(arr[0])):
            data.write(f'{di[j]}\t')
            for i in range(len(arr)):
                data.write(f'{arr[i][j]}\t')
            data.write('\n')

    print ("Channel 2")
    dfp = datapath('FlashDecay')
    path, filename = os.path.split(dfp)
    result = str.format("Filename: {0}; Flashes: {1}; Flash Mode: {2}; Flash-Power: {3}; Start: {4}; Window: {5}",filename, flash_count, high_power, flash_power, start_us, window_us)
    report(result, '.alpha.flashmap')
    result = str.format("HV1: {0}; HV2: {1}; DL1: {2}; DL2: {3}; Aperture-Pos.: {4}; GateDelay: {5}", hv1, hv2, dlstart, dlstop, as_pos, gatedelay)
    report(result, '.alpha.flashmap')
    print (filename)

    arr=[]
    di = []
    for i in range(window_count):
        di.append(start_us+i*window_us)
    for dl in range(dlstart, dlstop+dlstep, dlstep):
        await pmt1setdl(dl/256)
        await pmt2setdl(dl/256)
        await asyncio.sleep(delay)
        results = await flash_excitation_scan2(start_us, window_us, window_count, flash_count, flash_power, high_power, gatedelay)
        arr.append(results)
    await asx_close()
    await hvgate1off()
    await hvgate2off()
    await pmt1hvdisable()
    await pmt2hvdisable()
    with open(dfp, 'a') as data:
        for j in range(len(arr[0])):
            data.write(f'{di[j]}\t')
            for i in range(len(arr)):
                data.write(f'{arr[i][j]}\t')
            data.write('\n')
    report('Done', '.alpha.flashmap')
    print('flashmap Done')
    return (0)

async def flashrepeat(as_pos=1.0, iterations=10, dl1=0.3, dl2=0.28, hv1=0.5, hv2=0.5, start_us=0.0, window_us=1.0, window_count=100, flash_count=1, flash_power=0.5, high_power=1, gatedelay=1):
    delay = 0.1
    await pmt1setdl(dl1)
    await pmt2setdl(dl2)
    await pmt1sethv(hv1)
    await pmt2sethv(hv2)
    await pmt1hvenable()
    await pmt2hvenable()
    #await hvgate1on()
    #await hvgate2on()
    await asx(as_pos)
    await asyncio.sleep(delay)

    print ("Channel 1")
    dfp = datapath('FlashDecay')
    path, filename = os.path.split(dfp)
    result = str.format("Filename: {0}; Flashes: {1}; Flash Mode: {2}; Flash-Power: {3}; Start: {4}; Window: {5}",filename, flash_count, high_power, flash_power, start_us, window_us)
    report(result, '.alpha.flashmap')
    result = str.format("HV1: {0}; HV2: {1}; DL1: {2}; DL2: {3}; Aperture-Pos.: {4}; GateDelay: {5}; Iterations: {6}", hv1, hv2, dl1, dl2, as_pos, gatedelay, iterations)
    report(result, '.alpha.flashmap')
    print (filename)

    arr=[]
    di = []
    for i in range(window_count):
        di.append(start_us+i*window_us)
    for i in range(iterations):
        results = await flash_excitation_scan1(start_us, window_us, window_count, flash_count, flash_power, high_power, gatedelay)
        arr.append(results)
    with open(dfp, 'a') as data:
        for j in range(len(arr[0])):
            data.write(f'{di[j]}\t')
            for i in range(len(arr)):
                data.write(f'{arr[i][j]}\t')
            data.write('\n')

    print ("Channel 2")
    dfp = datapath('FlashDecay')
    path, filename = os.path.split(dfp)
    result = str.format("Filename: {0}; Flashes: {1}; Flash Mode: {2}; Flash-Power: {3}; Start: {4}; Window: {5}",filename, flash_count, high_power, flash_power, start_us, window_us)
    report(result, '.alpha.flashmap')
    result = str.format("HV1: {0}; HV2: {1}; DL1: {2}; DL2: {3}; Aperture-Pos.: {4}; GateDelay: {5}; Iterations: {6}", hv1, hv2, dl1, dl2, as_pos, gatedelay, iterations)
    report(result, '.alpha.flashmap')
    print (filename)

    arr=[]
    di = []
    for i in range(window_count):
        di.append(start_us+i*window_us)
    for i in range(iterations):
        results = await flash_excitation_scan2(start_us, window_us, window_count, flash_count, flash_power, high_power, gatedelay)
        arr.append(results)
    await asx_close()
    #await hvgate1off()
    #await hvgate2off()
    await pmt1hvdisable()
    await pmt2hvdisable()
    with open(dfp, 'a') as data:
        for j in range(len(arr[0])):
            data.write(f'{di[j]}\t')
            for i in range(len(arr)):
                data.write(f'{arr[i][j]}\t')
            data.write('\n')
    report('Done', '.alpha.flashmap')
    print('flashmap Done')
    return (0)

